package commands

import (
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"os/exec"
	"path/filepath"
	"strings"

	"github.com/mitchellh/go-homedir"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yatool"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yamake"
	"a.yandex-team.ru/security/tools/goat/internal/ya"
)

var installCmd = &cobra.Command{
	Use:   "install [flags] [target]",
	Short: "Install package into $GOPATH/bin",
	RunE:  doInstall,
}

func init() {
	RootCmd.AddCommand(installCmd)
}

func doInstall(cmd *cobra.Command, args []string) error {
	target, err := parseTarget(args)
	if err != nil {
		return err
	}

	yFile, err := os.Open(filepath.Join(target, "ya.make"))
	if err != nil {
		return xerrors.Errorf("failed to open ya.make in current dir: %v", err)
	}

	yaMake, err := yamake.Parse(yFile)
	if err != nil {
		return xerrors.Errorf("failed parse ya.make: %v", err)
	}

	if yaMake.TargetType != yamake.TargetGoProgram {
		return xerrors.New("target must be a GO_PROGRAM")
	}

	tmpDir, err := ioutil.TempDir("", "goat-install")
	if err != nil {
		return xerrors.Errorf("failed to create tmp dir: %v", err)
	}
	simplelog.Debug("tmp dir created", "path", tmpDir)

	defer func() {
		_ = os.RemoveAll(tmpDir)
		simplelog.Debug("tmp dir removed", "path", tmpDir)
	}()

	yaArgs := []string{
		"make", "-r",
		"--output", tmpDir,
		target,
	}

	yaMakeCmd := exec.Command(ya.Path(), yaArgs...)
	yaMakeCmd.Stdout = os.Stdout
	yaMakeCmd.Stderr = os.Stderr

	fmt.Println("  - build target")
	simplelog.Debug("run", "cmd", strings.Join(yaMakeCmd.Args, " "))
	if err = yaMakeCmd.Run(); err != nil {
		return xerrors.Errorf("failed to build: %v", err)
	}

	currentRoot, err := filepath.Abs(target)
	if err != nil {
		return xerrors.Errorf("current absolute path: %v", err)
	}

	arcadiaRoot, err := yatool.FindArcadiaRoot(currentRoot)
	if err != nil {
		return err
	}

	arcadiaPath, err := filepath.Rel(arcadiaRoot, currentRoot)
	if err != nil {
		return xerrors.Errorf("arcadia project path: %v", err)
	}
	simplelog.Debug("detected project path", "path", arcadiaPath)

	binaryPath, err := findToolBinary(filepath.Join(tmpDir, arcadiaPath))
	if err != nil {
		return xerrors.Errorf("failed to find tool binary path: %v", err)
	}
	simplelog.Debug("binary found", "path", arcadiaPath)

	binPath, err := binFolder()
	if err != nil {
		return err
	}

	targetBinaryPath := filepath.Join(binPath, filepath.Base(binaryPath))
	targetBinaryTmpPath := targetBinaryPath + ".tmp"
	simplelog.Debug("copy binary", "src", binaryPath, "dst", targetBinaryTmpPath)

	fmt.Println("  - copy target into destination")
	err = copyFile(binaryPath, targetBinaryTmpPath)
	if err != nil {
		return xerrors.Errorf("failed to copy from %s to %s: %v", binaryPath, binPath, err)
	}

	simplelog.Debug("rename binary", "src", targetBinaryTmpPath, "dst", targetBinaryPath)
	err = os.Rename(targetBinaryTmpPath, targetBinaryPath)
	if err != nil {
		return xerrors.Errorf("failed to rename from %s to %s: %v", targetBinaryTmpPath, targetBinaryPath, err)
	}

	return nil
}

func findToolBinary(targetPath string) (string, error) {
	files, err := ioutil.ReadDir(targetPath)
	if err != nil {
		return "", err
	}

	for _, f := range files {
		if f.Mode().IsRegular() && f.Mode()&0111 != 0 {
			return filepath.Join(targetPath, f.Name()), nil
		}
	}

	return "", xerrors.New("tool not found")
}

func copyFile(src, dst string) (err error) {
	copyFileContents := func(src, dst string) (err error) {
		in, err := os.Open(src)
		if err != nil {
			return
		}
		defer func() {
			_ = in.Close()
		}()

		out, err := os.Create(dst)
		if err != nil {
			return
		}
		defer func() {
			cerr := out.Close()
			if err == nil {
				err = cerr
			}
		}()

		if _, err = io.Copy(out, in); err != nil {
			return
		}
		err = out.Sync()
		return
	}

	sfi, err := os.Stat(src)
	if err != nil {
		return
	}

	if !sfi.Mode().IsRegular() {
		return xerrors.Errorf("non-regular source file %s (%q)", sfi.Name(), sfi.Mode().String())
	}

	dfi, err := os.Stat(dst)
	if err != nil {
		if !os.IsNotExist(err) {
			return
		}
	} else {
		if !(dfi.Mode().IsRegular()) {
			return xerrors.Errorf("non-regular destination file %s (%q)", dfi.Name(), dfi.Mode().String())
		}

		if os.SameFile(sfi, dfi) {
			return
		}
	}

	err = copyFileContents(src, dst)
	if err != nil {
		return
	}

	err = os.Chmod(dst, sfi.Mode())
	return
}

func binFolder() (string, error) {
	if goPath := os.Getenv("GOPATH"); goPath != "" {
		return filepath.Join(goPath, "bin"), nil
	}

	h, err := homedir.Dir()
	if err != nil {
		return "", fmt.Errorf("no ENV[GOPATH] and can't get user homedir: %w", err)
	}

	return filepath.Join(h, "go", "bin"), nil
}
