package commands

import (
	"context"
	"fmt"
	"os"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/sectools"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/tools/goat/internal/config"
	"a.yandex-team.ru/security/tools/goat/internal/project"
)

var RootCmd = &cobra.Command{
	Use:          "goat",
	SilenceUsage: true,
	Short:        "golang helper for Arcadia projects",
	Long: `Golang Arcadia Tools

Code&Doc: https://a.yandex-team.ru/arc/trunk/arcadia/security/tools/goat`,
	PersistentPreRun: func(cmd *cobra.Command, args []string) {
		if beVerbose {
			simplelog.SetLevel(simplelog.DebugLevel)
		}

		if cmd.Name() != "self-update" && !skipVersionCheck {
			checkVersion()
		}
	},
}

var (
	beVerbose        bool
	skipVersionCheck bool
	projectConfig    = project.LoadConfig()
)

func init() {
	flags := RootCmd.PersistentFlags()
	flags.BoolVarP(&beVerbose, "verbose", "v", false, "verbose output")
	flags.BoolVar(&skipVersionCheck, "skip-version-check", false, "skip version check")
}

func Execute() {
	RootCmd.Version = config.FullVersion()
	RootCmd.SetVersionTemplate("{{.Version}}\n")

	if err := RootCmd.Execute(); err != nil {
		os.Exit(1)
	}
}

func checkVersion() {
	ctx, cancel := context.WithTimeout(context.Background(), 1*time.Second)
	defer cancel()

	stClient := sectools.NewClient(config.ToolName)
	isLatest, latestVersion, err := stClient.IsLatestVersion(ctx, config.FullVersion())
	if !isLatest && err == nil {
		simplelog.Warn("----------")
		simplelog.Warn(fmt.Sprintf("New version available: %s", latestVersion))
		simplelog.Warn("Please update goat with 'goat self-update' or manually")
		simplelog.Warn("----------")
	}
}

func parseTarget(args []string) (target string, err error) {
	switch len(args) {
	case 0:
		target = "."
		return
	case 1:
		target = args[0]
		if _, err = os.Stat(target); err != nil {
			err = xerrors.Errorf("failed to check target: %w", err)
		}
		return
	default:
		err = xerrors.New("too many targets")
		return
	}
}
