package commands

import (
	"context"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/libs/go/sectools"
	"a.yandex-team.ru/security/tools/goat/internal/config"
)

var selfUpdateCmd = &cobra.Command{
	Use:   "self-update",
	Short: "Update goat to the latest version",
	RunE:  doSelfUpdate,
}

func init() {
	RootCmd.AddCommand(selfUpdateCmd)
}

func doSelfUpdate(_ *cobra.Command, _ []string) error {
	selfEx, err := os.Executable()
	if err != nil {
		return fmt.Errorf("failed to determine executable path: %w", err)
	}

	selfStat, err := os.Stat(selfEx)
	if err != nil {
		return fmt.Errorf("failed to determine executable path: %w", err)
	}

	dir, name := filepath.Split(selfEx)
	dstFile, err := ioutil.TempFile(dir, name+".tmp-")
	if err != nil {
		return err
	}

	defer func() {
		// cleanup
		_ = dstFile.Close()
		_ = os.Remove(dstFile.Name())
	}()

	stClient := sectools.NewClient(config.ToolName,
		sectools.WithCurrentVersion(config.FullVersion()),
		sectools.WithPreferFastURL(),
	)

	ctx, cancel := context.WithTimeout(context.Background(), 1*time.Minute)
	defer cancel()

	err = stClient.DownloadLatestVersion(ctx, dstFile)
	switch {
	case errors.Is(err, sectools.ErrDownloadSameVersion):
		fmt.Println("You already use latest version, nothing to do")
		return nil
	case err != nil:
		return fmt.Errorf("failed to request sec-tools service: %w", err)
	}

	if err := dstFile.Close(); err != nil {
		return err
	}

	if err := os.Chmod(dstFile.Name(), selfStat.Mode()); err != nil {
		return err
	}

	if err := os.Rename(dstFile.Name(), selfEx); err != nil {
		return err
	}

	fmt.Println("Goat updated")
	return nil
}
