package child

import (
	"context"
	"fmt"
	"os"
	"sync"
	"syscall"

	"github.com/go-cmd/cmd"

	"a.yandex-team.ru/library/go/core/xerrors"
)

var (
	ErrExited         = xerrors.NewSentinel("child exited")
	ErrNotStarted     = xerrors.NewSentinel("child not started")
	ErrAlreadyStarted = xerrors.NewSentinel("child already started")
)

type Child struct {
	name    string
	command string
	args    []string
	cmd     *cmd.Cmd
	mu      sync.Mutex
}

func NewChild(name, command string, args []string) *Child {
	child := &Child{
		name:    name,
		command: command,
		args:    args,
	}

	child.cmd = child.newCmd()
	return child
}

func (c *Child) Name() string {
	return c.name
}

func (c *Child) Respawn(ctx context.Context) error {
	_ = c.Stop()

	c.mu.Lock()
	c.cmd = c.newCmd()
	c.mu.Unlock()

	return c.Start(ctx)
}

func (c *Child) Start(ctx context.Context) error {
	c.mu.Lock()
	if c.isStarted() {
		c.mu.Unlock()
		return ErrAlreadyStarted.WithFrame()
	}

	statusChan := c.cmd.Start()
	c.mu.Unlock()

	// TODO(buglloc): WTF?!
	fmt.Printf("child %q started\n", c.name)
	for {
		select {
		case line := <-c.cmd.Stdout:
			fmt.Println(line)
		case line := <-c.cmd.Stderr:
			_, _ = fmt.Fprintln(os.Stderr, line)
		case <-statusChan:
			return ErrExited.WithFrame()
		case <-ctx.Done():
			_ = c.Stop()
		}
	}
}

func (c *Child) Status() Status {
	c.mu.Lock()
	defer c.mu.Unlock()

	status := c.cmd.Status()
	return Status{
		Started:        status.PID > 0,
		ExitExternally: !status.Complete,
		PID:            status.PID,
		Error:          status.Error,
		ExitCode:       status.Exit,
	}
}

func (c *Child) Ping() error {
	c.mu.Lock()
	defer c.mu.Unlock()

	pid := c.cmd.Status().PID
	if pid <= 0 {
		return ErrNotStarted.WithFrame()
	}

	process, err := os.FindProcess(pid)
	if err != nil {
		return xerrors.Errorf("can't find child process (%d): %w", pid, err)
	}

	err = process.Signal(syscall.Signal(0))
	if err != nil {
		return xerrors.Errorf("sending ping signal to child (%d) failed: %w", pid, err)
	}

	return nil
}

func (c *Child) Stop() error {
	c.mu.Lock()
	defer c.mu.Unlock()

	if !c.isStarted() {
		return ErrNotStarted.WithFrame()
	}

	return c.cmd.Stop()
}

func (c *Child) newCmd() *cmd.Cmd {
	cmdOptions := cmd.Options{
		Buffered:  false,
		Streaming: true,
	}

	return cmd.NewCmdOptions(cmdOptions, c.command, c.args...)
}

func (c *Child) isStarted() bool {
	return c.cmd.Status().PID > 0
}
