package commands

import (
	"context"
	"os"
	"os/signal"
	"sync"
	"syscall"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/tools/starter/internal/child"
	"a.yandex-team.ru/security/tools/starter/internal/server"
	"a.yandex-team.ru/security/tools/starter/pkg/starter"
)

var startCmd = &cobra.Command{
	Use:          "start",
	SilenceUsage: true,
	Short:        "start child",
	RunE: func(cmd *cobra.Command, args []string) error {
		if len(args) <= 0 {
			return xerrors.New("please provide binary for repo checks")
		}

		if isChildRunning(childName) {
			return xerrors.Errorf("child %s already running", childName)
		}

		wg := sync.WaitGroup{}
		wg.Add(2)

		//Setup logger
		logger := newLogger(childName)

		// Setup our child
		ch := child.NewChild(childName, args[0], args[1:])

		// Setup command server
		srv := server.New(ch, server.WithLogger(logger))

		// Starting control server
		ctx, cancel := context.WithCancel(context.Background())
		defer cancel()

		go func() {
			defer wg.Done()

			if err := srv.ListenAndServe(); err != nil {
				cancel()
				logger.Error("ctrl server stopped with error", log.Error(err))
			}
		}()

		exitCode := 0
		// Starting child
		go func() {
			defer wg.Done()
			// server must be stopped too!
			defer srv.Shutdown()

			err := ch.Start(ctx)
			if !xerrors.Is(err, child.ErrExited) {
				logger.Error("failed to start child", log.Error(err))
				exitCode = 1
				return
			}

			status := ch.Status()
			if status.ExitExternally {
				logger.Error("child externally terminated, server will be stopped",
					log.Int("exit_code", status.ExitCode),
				)
				exitCode = 1
				return
			}

			logger.Warn("child exited and will be respawned",
				log.Int("exit_code", status.ExitCode),
			)

			if status.ExitCode > 0 {
				exitCode = status.ExitCode
			}
		}()

		// Setup quit channel
		go func() {
			quit := make(chan os.Signal, 1)
			signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)

			// Wait termination or exit
			<-quit

			cancel()
		}()

		wg.Wait()
		os.Exit(exitCode)
		return nil
	},
}

func init() {
	rootCmd.AddCommand(startCmd)
}

func isChildRunning(childName string) bool {
	client, err := starter.NewClient(
		starter.WithChild(childName),
		starter.WithLogger(newLogger(childName)),
	)
	if err != nil {
		return false
	}

	ctx, cancel := context.WithTimeout(context.Background(), 1*time.Second)
	defer cancel()
	_, err = client.Ping(ctx)
	return err == nil
}
