package config

import (
	"context"
	"fmt"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/library/go/core/log"
)

type AppConfig struct {
	Addr               string        `config:"app_addr" toml:"addr" yaml:"addr"`
	ReadTimeout        time.Duration `config:"app_read_timeout" toml:"read_timeout" yaml:"read_timeout"`
	WriteTimeout       time.Duration `config:"app_write_timeout" toml:"write_timeout" yaml:"write_timeout"`
	IdleTimeout        time.Duration `config:"app_idle_timeout" toml:"idle_timeout" yaml:"idle_timeout"`
	ShutdownTimeout    time.Duration `config:"app_shutdown_timeout" toml:"shutdown_timeout" yaml:"shutdown_timeout"`
	MaxBodySize        int64         `config:"app_max_body_size" toml:"max_body_size" yaml:"max_body_size"`
	ReportWriteTimeout time.Duration `config:"app_report_write_timeout" toml:"report_write_timeout" yaml:"report_write_timeout"`
}

type Config struct {
	LogLevel string    `config:"log_level" toml:"log_level" yaml:"log_level"`
	App      AppConfig `config:"app" toml:"app" yaml:"app"`
}

func NewConfig(configPath string) (*Config, error) {
	var backends []backend.Backend
	if configPath != "" {
		backends = []backend.Backend{
			file.NewBackend(configPath),
			env.NewBackend(),
		}
	} else {
		backends = []backend.Backend{
			env.NewBackend(),
		}
	}
	loader := confita.NewLoader(backends...)

	cfg := &Config{
		LogLevel: log.InfoString,
		App: AppConfig{
			Addr:               "127.0.0.1:3000",
			ReadTimeout:        4 * time.Second,
			WriteTimeout:       3 * time.Second,
			IdleTimeout:        120 * time.Second,
			ReportWriteTimeout: 500 * time.Millisecond,
			ShutdownTimeout:    5 * time.Minute,
			MaxBodySize:        32 << 10,
		},
	}

	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, fmt.Errorf("failed to parse config: %w", err)
	}

	return cfg, nil
}
