package server

import (
	"context"
	"net"

	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/tools/starter/internal/child"
	"a.yandex-team.ru/security/tools/starter/internal/uds"
	"a.yandex-team.ru/security/tools/starter/pkg/starter/rpc"
)

type (
	Server struct {
		child *child.Child
		grpc  *grpc.Server
		l     log.Structured
	}
)

func New(child *child.Child, opts ...Option) *Server {
	s := &Server{
		child: child,
	}

	for _, opt := range opts {
		opt(s)
	}

	s.grpc = grpc.NewServer(grpc.UnaryInterceptor(s.errHandler))
	return s
}

func (s *Server) onStart() (err error) {
	return nil
}

func (s *Server) onEnd() (err error) {
	return nil
}

func (s *Server) ListenAndServe() error {
	err := s.onStart()
	if err != nil {
		return xerrors.Errorf("failed to start server: %w", err)
	}

	defer func() {
		err := s.onEnd()
		if err != nil {
			s.l.Error("failed to stop", log.Error(err))
		}
	}()

	addr := uds.Addr(s.child.Name())
	listener, err := net.Listen("unix", addr)
	if err != nil {
		return xerrors.Errorf("failed to listen: %w", err)
	}

	defer func() {
		_ = listener.Close()
	}()

	rpc.RegisterChildServer(s.grpc, s)

	s.l.Info("ctrl server started", log.String("addr", addr))
	return s.grpc.Serve(listener)
}

func (s *Server) Shutdown() {
	s.grpc.GracefulStop()
}

func (s *Server) errHandler(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo, handler grpc.UnaryHandler) (interface{}, error) {
	resp, err := handler(ctx, req)
	if err != nil {
		s.l.Error("request error", log.String("method", info.FullMethod), log.Error(err))
	}
	return resp, err
}
