package starter

import (
	"context"

	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/tools/starter/internal/uds"
	"a.yandex-team.ru/security/tools/starter/pkg/starter/rpc"
)

type Client struct {
	addr     string
	l        log.Structured
	conn     *grpc.ClientConn
	child    rpc.ChildClient
	dialOpts []grpc.DialOption
}

func NewClient(opts ...Option) (client *Client, err error) {
	client = &Client{
		addr: uds.Addr("unnamed"),
		l:    &nop.Logger{},
	}

	for _, opt := range opts {
		opt(client)
	}

	client.conn, err = NewConnection(client.addr, client.dialOpts...)
	if err != nil {
		return nil, xerrors.Errorf("failed to create gRPC connection: %w", err)
	}
	client.child = rpc.NewChildClient(client.conn)
	return client, nil
}

func (c *Client) Close() error {
	if err := c.conn.Close(); err != nil {
		return xerrors.Errorf("failed to close gRPC connection: %w", err)
	}
	return nil
}

func (c *Client) Ping(ctx context.Context) (*PingReply, error) {
	rsp, err := c.child.Ping(ctx, &rpc.PingRequest{})
	if err != nil {
		return nil, err
	}

	return &PingReply{
		Status: rsp.Status,
		Reason: rsp.Reason,
	}, nil
}

func (c *Client) Stop(ctx context.Context) error {
	_, err := c.child.Stop(ctx, &rpc.StopRequest{})
	return err
}
