package main

import (
	"context"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/vt-proxy/internal/app"
	"a.yandex-team.ru/security/vt-proxy/internal/filereport"
	"a.yandex-team.ru/security/vt-proxy/internal/logger"
)

const (
	shutdownTimeout = 10 * time.Second
)

func main() {
	maxprocs.AdjustYP()

	cfg := app.Config{
		ListenAddr: ":80",
		FileReport: filereport.Config{
			VtAuthToken:  os.Getenv("VT_TOKEN"),
			YdbAuthToken: os.Getenv("YDB_TOKEN"),
			YdbDatabase:  os.Getenv("YDB_DATABASE"),
			YdbEndpoint:  os.Getenv("YDB_ENDPOINT"),
		},
	}

	application := app.New(cfg)

	defer func() {
		if err := application.Close(); err != nil {
			logger.Error("failed to close server", log.Error(err))
		}
	}()

	// Setup quit channel
	quit := make(chan os.Signal, 2)

	// Start server
	go func() {
		if err := application.ListenAndServe(); err != nil {
			logger.Error("server stopped", log.Error(err))
			// send signal manually, server failed to start
			signal.Stop(quit)
			quit <- syscall.SIGTERM
		}
	}()

	// Wait termination
	signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
	<-quit

	// shut down
	ctx, cancel := context.WithTimeout(context.Background(), shutdownTimeout)
	defer cancel()

	if err := application.Shutdown(ctx); err != nil {
		logger.Error("server shutdown failed", log.Error(err))
	}
}
