package app

import (
	"context"
	"net/http"
	"regexp"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/vt-proxy/internal/filereport"
	"a.yandex-team.ru/security/vt-proxy/internal/unistat"
)

type Config struct {
	FileReport filereport.Config
	ListenAddr string
}

type App struct {
	cfg        Config
	ctx        context.Context
	shutdownFn context.CancelFunc
	echo       *echo.Echo
	reporter   *filereport.ReportClient
}

func New(cfg Config) *App {
	ctx, cancel := context.WithCancel(context.Background())
	return &App{
		cfg:        cfg,
		ctx:        ctx,
		shutdownFn: cancel,
		echo:       echo.New(),
	}
}

func (a *App) onStart() (err error) {
	a.reporter, err = filereport.NewClient(a.ctx, a.cfg.FileReport)
	if err != nil {
		return xerrors.Errorf("failed to create filereport client: %w", err)
	}

	return nil
}

func (a *App) onEnd() error {
	a.shutdownFn()
	return nil
}

func (a *App) Close() error {
	a.reporter.Close()
	return nil
}

func (a *App) ListenAndServe() error {
	err := a.onStart()
	if err != nil {
		return xerrors.Errorf("failed to start server: %w", err)
	}

	defer func() {
		err := a.onEnd()
		if err != nil {
			simplelog.Error("failed to stop", "err", err)
		}
	}()

	a.echo.Use(middleware.RecoverWithConfig(middleware.RecoverConfig{
		Skipper:           middleware.DefaultSkipper,
		StackSize:         4 << 10, // 4 KB
		DisableStackAll:   true,
		DisablePrintStack: false,
	}))

	stat := unistat.NewCounter()
	sha256Re := regexp.MustCompile(`^[a-f0-9]{64}$`)

	a.echo.GET("/ping", func(c echo.Context) error {
		return c.String(200, "pong")
	})

	a.echo.GET("/version", func(c echo.Context) error {
		return c.String(200, buildinfo.Info.ProgramVersion)
	})

	a.echo.GET("/unistat", func(c echo.Context) error {
		return c.JSON(http.StatusOK, stat.FlushSignals())
	})

	a.echo.GET("/api/v1/updated/today", func(c echo.Context) error {
		count, err := a.reporter.UpdatedToday()
		if err != nil {
			return returnErr(c, err)
		}

		return c.JSON(http.StatusOK, echo.Map{
			"count": count,
		})
	})

	a.echo.GET("/api/v1/file/report/:hash", func(c echo.Context) error {
		hash := c.Param("hash")
		if !sha256Re.Match([]byte(hash)) {
			return returnErr(c, xerrors.New("hash doesn't look like sha256"))
		}

		var options ReportOptions
		if err := c.Bind(&options); err != nil {
			return returnErr(c, err)
		}

		oldReport, report, cacheStatus, err := a.reporter.GetReport(hash, options.Force)
		if err != nil {
			return returnErr(c, err)
		}

		if report == nil {
			return returnErr(c, xerrors.New("no report retrieved"))
		}

		cacheStatus.Stat(stat)
		result := ReportResponse{
			CacheStatus: cacheStatus,
			Found:       report.Found,
			Md5:         report.Md5,
			Sha1:        report.Sha1,
			Sha256:      report.Sha256,
			Positives:   report.Positives,
			Total:       report.Total,
			CreatedAt:   &report.CreatedAt,
			UpdatedAt:   &report.UpdatedAt,
			Scans:       report.Scans,
		}

		if oldReport != nil && oldReport.Positives != report.Positives {
			result.CacheStatus.Status = filereport.CacheStatusVtChanged
			result.DiffPositives = report.Positives - oldReport.Positives
			result.DiffUpdatedAt = &oldReport.UpdatedAt
		}

		return returnOk(c, result)
	})

	return a.echo.Start(a.cfg.ListenAddr)
}

func (a *App) Shutdown(ctx context.Context) error {
	return a.echo.Shutdown(ctx)
}
