package unistat

import "sync/atomic"

/*
* documentation for stats semantic: https://wiki.yandex-team.ru/golovan/stat-handle/#defoltnyesigopt-suffiksy
 */
type Counter struct {
	CacheMiss       uint64
	CacheHit        uint64
	CacheInvalidate uint64
	VtFetch         uint64
	VtRefresh       uint64
	VtNew           uint64
	VtFail          uint64
}

func NewCounter() *Counter {
	return &Counter{}
}

func (c *Counter) AddCacheMiss() {
	atomic.AddUint64(&c.CacheMiss, 1)
}

func (c *Counter) AddCacheHit() {
	atomic.AddUint64(&c.CacheHit, 1)
}

func (c *Counter) AddCacheInvalidate() {
	atomic.AddUint64(&c.CacheInvalidate, 1)
}

func (c *Counter) AddVtRefresh() {
	atomic.AddUint64(&c.VtRefresh, 1)
}

func (c *Counter) AddVtNew() {
	atomic.AddUint64(&c.VtNew, 1)
}

func (c *Counter) AddVtFetch() {
	atomic.AddUint64(&c.VtFetch, 1)
}

func (c *Counter) AddVtFail() {
	atomic.AddUint64(&c.VtFail, 1)
}

func (c *Counter) Flush() Counter {
	return Counter{
		CacheMiss:       atomic.SwapUint64(&c.CacheMiss, 0),
		CacheHit:        atomic.SwapUint64(&c.CacheHit, 0),
		CacheInvalidate: atomic.SwapUint64(&c.CacheInvalidate, 0),
		VtRefresh:       atomic.SwapUint64(&c.VtRefresh, 0),
		VtNew:           atomic.SwapUint64(&c.VtNew, 0),
		VtFail:          atomic.SwapUint64(&c.VtFail, 0),
	}
}

func (c *Counter) FlushSignals() [][]interface{} {
	return [][]interface{}{
		{"cache-miss_summ", atomic.SwapUint64(&c.CacheMiss, 0)},
		{"cache-hit_summ", atomic.SwapUint64(&c.CacheHit, 0)},
		{"cache-invalidate_summ", atomic.SwapUint64(&c.CacheInvalidate, 0)},
		{"vt-fetch_summ", atomic.SwapUint64(&c.VtFetch, 0)},
		{"vt-refresh_summ", atomic.SwapUint64(&c.VtRefresh, 0)},
		{"vt-new_summ", atomic.SwapUint64(&c.VtNew, 0)},
		{"vt-fail_summ", atomic.SwapUint64(&c.VtFail, 0)},
	}
}
