package virustotal

import (
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/vt-proxy/internal/logger"
)

const (
	upstreamURI    = "https://virustotal.viewer.yandex-team.ru/vtapi/v2/file/report?"
	retryCount     = 2
	requestTimeout = 2 * time.Second
)

type (
	VtStatus struct {
		ResponseCode int    `json:"response_code"`
		VerboseMsg   string `json:"verbose_msg"`
	}

	VtFileReport struct {
		VtStatus
		Found     bool   `json:"-"`
		Md5       string `json:"md5"`
		Sha1      string `json:"sha1"`
		Sha256    string `json:"sha256"`
		ScanDate  string `json:"scan_date"`
		Positives int    `json:"positives"`
		Total     int    `json:"total"`
		Scans     map[string]struct {
			Detected bool   `json:"detected"`
			Version  string `json:"version"`
			Result   string `json:"result"`
			Update   string `json:"update"`
		} `json:"scans"`
	}

	ClientConfig struct {
		AuthToken string
	}

	Client struct {
		httpc *resty.Client
	}
)

var (
	ErrRateLimitExceeded = xerrors.NewSentinel("request rate limit exceeded")
)

func NewClient(cfg ClientConfig) *Client {
	httpc := resty.New().
		SetHeader("Authorization", "OAuth "+cfg.AuthToken).
		SetRetryCount(retryCount).
		SetTimeout(requestTimeout).
		SetLogger(logger.Logger().Fmt()).
		SetBaseURL(upstreamURI)

	return &Client{
		httpc: httpc,
	}
}
func (c *Client) FileReport(resource string) (*VtFileReport, error) {
	result := VtFileReport{}
	rsp, err := c.httpc.R().
		SetQueryParam("resource", resource).
		SetResult(&result).
		Get("")

	if err != nil {
		return nil, xerrors.Errorf("failed to request virus-total: %w", err)
	}

	switch rsp.StatusCode() {
	case 200:
		// https://www.virustotal.com/ru/documentation/public-api/#response-basics
		// response_code: if the item you searched for was not present in VirusTotal's dataset this result will be 0. If the requested item is still queued for analysis it will be -2. If the item was indeed present and it could be retrieved it will be 1. Any other case is detailed in the following sections.
		result.Found = result.ResponseCode == 1
		return &result, nil
	case 204:
		return nil, ErrRateLimitExceeded.WithFrame()
	default:
		return nil, xerrors.Errorf("vt return not 200 status code: %d", rsp.StatusCode())
	}
}
