package main

import (
	"context"
	"os"
	"sync"
	"time"

	"github.com/labstack/echo/v4"
	echoMiddleware "github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/waffles/middleware"
	"a.yandex-team.ru/security/waffles/pkg/config"
	"a.yandex-team.ru/security/waffles/pkg/dns"
	"a.yandex-team.ru/security/waffles/pkg/logger"
)

const (
	syncDuration = 2 * time.Hour
)

var (
	hostsTree *dns.Tree
	m         sync.RWMutex
)

func syncDNS() {
	logger.L.Info("cron job started")
	defer logger.L.Info("cron job finished")

	records, err := dns.PullDNSZone()
	if err != nil {
		logger.L.Error("failed to pull DNS zone", log.Error(err))
		return
	}

	tmp := dns.ParseDNSZone(records)
	m.Lock()
	hostsTree = tmp
	m.Unlock()
}

func GetIP(c echo.Context) error {
	ip := c.Param("ip")

	m.RLock()
	hosts, err := hostsTree.Hosts(ip)
	m.RUnlock()

	if err != nil {
		return err
	}

	return c.JSON(200, echo.Map{ip: hosts})
}

func newEcho(cfg *config.Config) *echo.Echo {
	e := echo.New()
	e.Debug = cfg.Debug

	e.Use(func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			req := c.Request()
			res := c.Response()
			start := time.Now()
			if err = next(c); err != nil {
				c.Error(err)
			}
			stop := time.Now()

			logger.L.Info("request",
				log.String("method", req.Method),
				log.String("uri", req.RequestURI),
				log.Int("statuc", res.Status),
				log.Duration("duration", stop.Sub(start)),
			)

			return nil
		}
	})

	e.Use(echoMiddleware.RecoverWithConfig(echoMiddleware.RecoverConfig{
		Skipper:           echoMiddleware.DefaultSkipper,
		StackSize:         4 << 10, // 4 KB
		DisableStackAll:   true,
		DisablePrintStack: false,
	}))

	e.GET("/ping", func(c echo.Context) error {
		return c.String(200, "pong")

	})

	v1 := e.Group("/api/v1")
	v1.Use(middleware.TVMMiddleware(middleware.TVMOptions{
		Services: middleware.TVMServices{
			cfg.AppTVMId: cfg.AppTVMName,
			2002104:      "Debby",
			2010214:      "Molly",
			2010090:      "C3P0",
		},
	}))
	v1.GET("/virtual_host/:ip", GetIP)

	user := e.Group("/user")
	user.Use(middleware.AuthMiddleware(cfg.ACL))
	user.GET("/virtual_host/:ip", GetIP)

	return e
}

func main() {
	logger.L.Info("started")
	maxprocs.AdjustYP()

	port := os.Getenv("QLOUD_HTTP_PORT")
	if port == "" {
		port = "3000"
	}

	cfg := config.Config{
		Port:    port,
		NodeEnv: os.Getenv("NODE_ENV"),
		AllowedLogins: []string{
			"a-abakumov",
			"procenkoeg",
			"ilyaon",
			"ezaitov",
			"buglloc",
			"horus",
			"gots",
			"shikari",
			"shashial",
			"limetime",
			"spellanser",
			"anton-k",
			"avkov",
			"ybuchnev",
			"kaleda",
			"velavokr", // Группа балансера
			"melkikh",
			"avasite",
			"dbeltukov",
			"ezzer",
		},
	}

	cfg.RefreshACL()
	if cfg.NodeEnv == "production" {
		cfg.Debug = false
		cfg.AppTVMId = 2008801
		cfg.AppTVMName = "Waffles Prod"
	} else if cfg.NodeEnv == "dev" {
		cfg.Debug = true
		cfg.AppTVMId = 2010372
		cfg.AppTVMName = "Waffles Dev"
	}

	logger.L.Info("pull DNS")
	records, err := dns.PullDNSZone()
	if err != nil {
		panic(err)
	}

	logger.L.Info("parse DNS")
	hostsTree = dns.ParseDNSZone(records)

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	go func() {
		nextCall := func() time.Duration {
			return time.Until(
				time.Now().Add(syncDuration).Round(syncDuration),
			)
		}

		timer := time.NewTimer(nextCall())
		defer timer.Stop()

		for {
			select {
			case <-ctx.Done():
				return
			case <-timer.C:
				syncDNS()
				timer.Reset(nextCall())
			}
		}
	}()

	logger.L.Info("setup HTTP server")
	e := newEcho(&cfg)
	e.Server.ReadTimeout = 5 * time.Second
	e.Server.WriteTimeout = 10 * time.Second

	logger.L.Info("start HTTP server")
	if err := e.Start(":" + port); err != nil {
		panic(err)
	}
}
