package dns

import (
	"bufio"
	"bytes"
	"compress/gzip"
	"io"
	"io/ioutil"
	"net/http"
	"os"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/libs/go/yahttp"
	"a.yandex-team.ru/security/waffles/pkg/logger"
)

const (
	dnsAPIEndpoint = "https://dns-api.yandex.net/v2.0/security-service/zones-content?contentDirectZones"
)

var (
	authToken  = os.Getenv("AUTH_SHARED_KEY")
	httpClient = yahttp.NewClient(yahttp.Config{
		RedirectPolicy: yahttp.RedirectNoFollow,
		DialTimeout:    10 * time.Second,
		Timeout:        600 * time.Second,
	})
)

func PullDNSZone() ([]Record, error) {
	req, _ := http.NewRequest("GET", dnsAPIEndpoint, nil)
	req.Header.Add("Accept", "text/plain")
	req.Header.Add("Accept-Encoding", "gzip")
	req.Header.Add("X-Auth-SharedKey", authToken)

	resp, err := httpClient.Do(req)
	if err != nil {
		logger.L.Error("Failed to do request to DNS API", log.Error(err))
		return nil, err
	}

	body, err := gzip.NewReader(resp.Body)
	if err != nil {
		logger.L.Error("Failed to do decompress from DNS API", log.Error(err))
		return nil, err
	}
	defer safeClose(body)

	data, err := ioutil.ReadAll(body)
	if err != nil {
		logger.L.Error("Failed to read response  from DNS API", log.Error(err))
		return nil, err
	}

	safeClose(resp.Body)

	scanner := bufio.NewScanner(bytes.NewReader(data))
	var result []Record
	for scanner.Scan() {
		line := strings.TrimSpace(scanner.Text())
		if line == "" || strings.HasPrefix(line, "#") {
			continue
		}

		parts := strings.Fields(line)
		// parts[0] - name
		// parts[1] - ttl
		// parts[2] - record_class
		// parts[3] - record_type
		// parts[4:] - record_data

		if len(parts) < 5 {
			logger.L.Error("invalid DNS record line", log.String("line", line))
			continue
		}

		ttl, err := strconv.ParseUint(parts[1], 10, 64)
		if err != nil {
			logger.L.Error("invalid DNS record TTL", log.String("ttl", parts[1]), log.String("line", line))
			continue
		}

		data := strings.ReplaceAll(
			strings.TrimSuffix(
				strings.Join(parts[4:], " "),
				".",
			),
			`"`, "",
		)

		result = append(result, Record{
			Name:  strings.TrimSuffix(parts[0], "."),
			TTL:   ttl,
			Class: parts[2],
			Type:  parts[3],
			Data:  data,
		})
	}

	return result, nil
}

func safeClose(closer io.Closer) {
	err := closer.Close()
	if err != nil {
		logger.L.Error("close failed", log.Error(err))
	}
}
