package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"path/filepath"
	"sync"

	"golang.org/x/sync/errgroup"
	"google.golang.org/protobuf/encoding/protojson"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/libs/go/porto"
	"a.yandex-team.ru/security/xray/internal/servers/worker/inspect"
	"a.yandex-team.ru/security/xray/internal/storage/layerstorage"
	"a.yandex-team.ru/security/xray/internal/storage/resstorage"
	"a.yandex-team.ru/security/xray/pkg/checks"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/yp/go/yp"
	"a.yandex-team.ru/yp/go/yson/ypapi"
)

const concurrency = 48

var checkTyp, workdir, checksDir, stagesFilter string
var logger log.Logger

func fatalf(msg string, a ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, "examinator: "+msg+"\n", a...)
	os.Exit(1)
}

func main() {
	maxprocs.AdjustAuto()
	flag.StringVar(&checkTyp, "check", "", "check to run")
	flag.StringVar(&workdir, "workdir", ".", "workdir to use")
	flag.StringVar(&checksDir, "checks-dir", "./checks", "dir with checks binaries")
	flag.StringVar(&stagesFilter, "filter", "", "stages filter")
	flag.Parse()

	zConfig := zap.ConsoleConfig(log.InfoLevel)
	zConfig.OutputPaths = []string{"stderr"}
	zlog, err := zap.New(zConfig)
	if err != nil {
		panic(err)
	}
	logger = zlog

	workdir, err = filepath.Abs(workdir)
	if err != nil {
		fatalf("can't normalize workdir: %v", err)
	}

	checksDir, err = filepath.Abs(checksDir)
	if err != nil {
		fatalf("can't normalize checks-dir: %v", err)
	}

	toInspect := make(chan *ypapi.TStage, 16384)
	g, ctx := errgroup.WithContext(context.Background())
	g.Go(func() error {
		return fetchStages(ctx, toInspect)
	})

	g.Go(func() error {
		return inspectStages(ctx, toInspect)
	})

	if err := g.Wait(); err != nil {
		fatalf("oops: %v", err)
	}
}

func fetchStages(ctx context.Context, toInspect chan<- *ypapi.TStage) error {
	ypc, err := yp.NewClient("xdc", yp.WithSystemAuthToken())
	if err != nil {
		return fmt.Errorf("create YP client: %w", err)
	}
	defer ypc.Close()

	const ypLimit = 100
	var continuationToken string
	for {
		rsp, err := ypc.SelectStages(ctx, yp.SelectStagesRequest{
			Format:            yp.PayloadFormatYson,
			Limit:             ypLimit,
			Filter:            stagesFilter,
			ContinuationToken: continuationToken,
			Selectors:         []string{"/spec", "/meta"},
		})
		if err != nil {
			return fmt.Errorf("request stages: %s", err)
		}

		for rsp.Next() {
			stageInfo := &ypapi.TStage{
				Spec: new(ypapi.TStageSpec),
				Meta: new(ypapi.TStageMeta),
			}
			if err = rsp.Fill(stageInfo.Spec, stageInfo.Meta); err != nil {
				logger.Error("failed to get stage info", log.Error(err))
				continue
			}

			logger.Info("schedule stage", log.String("fqid", stageInfo.GetMeta().GetFqid()))
			toInspect <- stageInfo
		}

		continuationToken = rsp.ContinuationToken()
		if continuationToken == "" {
			break
		}

		if rsp.Count() < ypLimit {
			break
		}
	}

	close(toInspect)
	return nil
}

func inspectStages(ctx context.Context, toInspect <-chan *ypapi.TStage) error {
	layerStorage, err := layerstorage.NewStorage(
		filepath.Join(workdir, "layers"),
		layerstorage.WithMaxSize(50*1024*1024*1024),
		layerstorage.WithLogger(logger),
	)
	if err != nil {
		return fmt.Errorf("create layer storage: %w", err)
	}
	defer layerStorage.Close()

	resStorage, err := resstorage.NewStorage(
		filepath.Join(workdir, "resources"),
		resstorage.WithMaxSize(50*1024*1024*1024),
		resstorage.WithLogger(logger),
	)
	if err != nil {
		return fmt.Errorf("create resource storage: %w", err)
	}
	defer resStorage.Close()

	portoc, err := porto.NewAPI(&porto.APIOpts{
		MaxConnections: concurrency * 2,
	})
	if err != nil {
		return fmt.Errorf("create porto client: %w", err)
	}
	defer portoc.Close()

	targetChecks, err := checks.NewCheck(checkTyp, check.Config{
		ContainerDir: "/checks",
		HostDir:      checksDir,
		AuthToken:    os.Getenv("CHECKS_TOKEN"),
	})
	if err != nil {
		return err
	}

	portoOpts := map[string]string{
		"enable_porto": "none",
		"isolate":      "true",
		// disable resolv.conf to resolve unexpected results in container
		"resolv_conf":  "",
		"user":         "root",
		"group":        "root",
		"memory_limit": "1024Mb",
		"bind":         fmt.Sprintf("%s /checks ro", checksDir),
	}

	wg := sync.WaitGroup{}
	for i := 0; i < concurrency; i++ {
		wg.Add(1)
		go func() {
			defer wg.Done()

			for {
				var stage *ypapi.TStage
				select {
				case <-ctx.Done():
					return
				case s, ok := <-toInspect:
					if !ok {
						return
					}
					stage = s
				}

				logger.Info("analyze stage", log.String("fqid", stage.GetMeta().GetFqid()))
				err := func() error {
					inspector := inspect.NewInspector(
						inspect.WithLayerStorage(layerStorage),
						inspect.WithResourcesStorage(resStorage),
						inspect.WithPortoAPI(portoc),
						inspect.WithChecks(targetChecks),
						inspect.WithLogger(logger),
						inspect.WithOutputLogger(&nop.Logger{}),
						inspect.WithAnalyzeID(stage.GetMeta().GetUuid()),
						inspect.WithWorkDir(filepath.Join(workdir, "inspector")),
						inspect.WithPortoOpts(portoOpts),
					)
					defer inspector.Close()

					results, err := inspector.Inspect(ctx, stage)
					if err != nil {
						return err
					}

					if len(results.Issues) == 0 {
						logger.Info("no issues", log.String("fqid", stage.GetMeta().GetFqid()))
						return nil
					}

					out, err := protojson.Marshal(results)
					if err != nil {
						return fmt.Errorf("failed to marshal results: %w", err)
					}

					_, _ = os.Stdout.WriteString(`{"stage":"`)
					_, _ = os.Stdout.WriteString(stage.GetMeta().GetFqid())
					_, _ = os.Stdout.WriteString(`", "results":`)
					_, _ = os.Stdout.Write(out)
					_, _ = os.Stdout.WriteString("}\n")
					return nil
				}()

				logger.Info("stage analyzed", log.String("fqid", stage.GetMeta().GetFqid()))
				if err != nil {
					logger.Error("failed to check stage", log.String("fqid", stage.GetMeta().GetFqid()), log.Error(err))
				}
			}
		}()
	}

	wg.Wait()
	return nil
}
