package main

import (
	"flag"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/xray/internal/servers/grpc"
	"a.yandex-team.ru/security/xray/internal/servers/grpc/config"
	"a.yandex-team.ru/security/xray/internal/servers/grpc/infra"
)

func main() {
	maxprocs.AdjustAuto()

	var cfgPath string
	flag.StringVar(&cfgPath, "config", "", "path to config")
	flag.Parse()

	cfg, err := config.Load(cfgPath)
	if err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "unable to load config: %v", err)
		os.Exit(1)
	}

	// Setup new API server
	i := infra.New(cfg)
	srv, err := grpc.New(i)
	if err != nil {
		panic(err)
	}

	stopChan := make(chan os.Signal, 1)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)

	errChan := make(chan error, 1)
	go func() {
		if err := srv.ListenAndServe(); err != nil {
			errChan <- err
		}
	}()

	select {
	case <-stopChan:
		// TODO(buglloc): timeout
		srv.GracefulStop()
	case err := <-errChan:
		i.Logger.Error("unexpected api error", log.Error(err))
	}
}
