package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	_ "a.yandex-team.ru/security/xray/assets/humanizer"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/config"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/infra"
)

const shutdownDeadline = 5 * time.Second

func main() {
	maxprocs.AdjustAuto()

	var cfgPath string
	flag.StringVar(&cfgPath, "config", "", "path to config")
	flag.Parse()

	cfg, err := config.Load(cfgPath)
	if err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "unable to load config: %v", err)
		os.Exit(1)
	}

	// Setup new infra
	i := infra.New(cfg)
	// Setup new HTTP server
	srv := humanizer.New(i)

	stopChan := make(chan os.Signal, 1)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)

	errChan := make(chan error, 1)
	go func() {
		if err := srv.ListenAndServe(); err != nil {
			errChan <- err
		}
	}()

	select {
	case <-stopChan:
		ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
		defer cancel()

		if err := srv.Shutdown(ctx); err != nil {
			i.Logger.Error("failed to shutdown watcher", log.Error(err))
		}
	case err := <-errChan:
		i.Logger.Error("unexpected humanizer error", log.Error(err))
	}
}
