package main

import (
	"context"
	"flag"
	"fmt"
	"io"
	"os"

	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/security/xray/pkg/xray"
	"a.yandex-team.ru/security/xray/pkg/xrayerrors"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/yp/go/yp"
)

const (
	stagesLimit = 50
)

type (
	stageInfo struct {
		Revision  uint32
		AccountID string
		ID        string
		UUID      string
	}
)

var (
	force     bool
	maxStages int
)

func errorf(format string, a ...interface{}) {
	_, _ = os.Stderr.WriteString("xray-scheduler: " + fmt.Sprintf(format, a...) + "\n")
	os.Exit(1)
}

func main() {
	flag.BoolVar(&force, "force", false, "ignore latest analysis data")
	flag.IntVar(&maxStages, "max-stages", 0, "maximum stages count to schedule")
	flag.Parse()

	zlog, err := zap.NewDeployLogger(log.DebugLevel)
	if err != nil {
		errorf("failed to create logger: %s", err)
	}

	xrayClient, err := xray.NewClient(
		xray.WithTokenAuth(os.Getenv("XRAY_TOKEN")),
		xray.WithLogger(zlog),
	)
	if err != nil {
		errorf("failed to create xray client: %s", err)
	}

	defer checkClose(xrayClient)

	ypClient, err := yp.NewClient(
		"xdc",
		yp.WithSystemAuthToken(),
		yp.WithLogger(zlog),
	)
	if err != nil {
		errorf("failed to create yp client: %s", err)
	}

	defer ypClient.Close()

	var (
		continuationToken string
		stagesScheduled   int
	)
	for {
		stages, err := ypClient.SelectStages(context.Background(), yp.SelectStagesRequest{
			Format:            yp.PayloadFormatYson,
			Selectors:         []string{"/spec/revision", "/spec/account_id", "/meta/id", "/meta/uuid"},
			Limit:             stagesLimit,
			ContinuationToken: continuationToken,
			Filter:            `[/status/validated/status] = "true"`,
		})
		if err != nil {
			errorf("failed to request stages: %s", err)
		}

		var stage stageInfo
		for stages.Next() {
			if err := stages.Fill(&stage.Revision, &stage.AccountID, &stage.ID, &stage.UUID); err != nil {
				zlog.Errorf("failed to get stage info: %s", err)
				continue
			}

			if err := scheduleStage(xrayClient, stage); err != nil {
				if status.Code(err) == xrayerrors.ErrCodeConflictSchedule {
					zlog.Infof(
						"skip already analyzed stage: stage_id=%q, stage_uuid=%q, stage_revision=%d",
						stage.ID, stage.UUID, stage.Revision,
					)
				} else {
					zlog.Errorf("failed to schedule stage analysis: %s", err)
				}
				continue
			}

			stagesScheduled++
			zlog.Infof(
				"stage scheduled: stage_id=%q, stage_uuid=%q, stage_revision=%d",
				stage.ID, stage.UUID, stage.Revision,
			)
		}

		if maxStages > 0 && stagesScheduled >= maxStages {
			break
		}

		continuationToken = stages.ContinuationToken()
		if continuationToken == "" {
			break
		}

		if stages.Count() < stagesLimit {
			break
		}
	}

	zlog.Infof("scheduled %d stages", stagesScheduled)
}

func checkClose(closer io.Closer) {
	if err := closer.Close(); err != nil {
		errorf("close failed: %s", err)
	}
}

func scheduleStage(xrayClient *xray.Client, stage stageInfo) error {
	_, err := xrayClient.Schedule(context.Background(), &xrayrpc.StageScheduleRequest{
		Stage: &xrayrpc.Stage{
			Id:       stage.ID,
			Uuid:     stage.UUID,
			Revision: stage.Revision,
		},
		Force: force,
	})
	return err
}
