package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/xray/internal/servers/watcher"
	"a.yandex-team.ru/security/xray/internal/servers/watcher/config"
)

const shutdownDeadline = time.Second * 300

func logger(cfg *config.Config) log.Logger {
	zlog, err := zap.NewDeployLogger(cfg.LogLvl.Level)
	if err != nil {
		panic(err)
	}

	return zlog
}

func main() {
	maxprocs.AdjustAuto()

	var cfgPath string
	flag.StringVar(&cfgPath, "config", "", "path to config")
	flag.Parse()

	cfg, err := config.Load(cfgPath)
	if err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "unable to load config: %v", err)
		os.Exit(1)
	}

	zlog := logger(cfg)
	w, err := watcher.NewWatcher(cfg, zlog)
	if err != nil {
		panic(err)
	}

	stopChan := make(chan os.Signal, 1)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)

	errChan := make(chan error, 1)
	go func() {
		if err := w.Start(); err != nil {
			errChan <- err
		}
	}()

	select {
	case <-stopChan:
		ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
		defer cancel()

		if err := w.Shutdown(ctx); err != nil {
			zlog.Error("failed to shutdown watcher", log.Error(err))
		}
	case err := <-errChan:
		zlog.Error("unexpected watcher error", log.Error(err))
	}
}
