package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/xray/internal/servers/worker"
	"a.yandex-team.ru/security/xray/internal/servers/worker/config"
)

func logger(cfg *config.Config) log.Logger {
	zlog, err := zap.NewDeployLogger(cfg.LogLvl.Level)
	if err != nil {
		panic(err)
	}

	return zlog
}

func prepareDirs(cfg *config.Config) {
	toCreate := []string{
		cfg.Worker.WorkDir,
		cfg.Worker.LayerStorage.Dir,
		cfg.Worker.ResourceStorage.Dir,
	}

	for _, p := range toCreate {
		if err := os.MkdirAll(p, 0755); err != nil {
			panic(fmt.Sprintf("create dir %q: %v", p, err))
		}
	}
}

func main() {
	maxprocs.AdjustAuto()

	var cfgPath string
	flag.StringVar(&cfgPath, "config", "", "path to config")
	flag.Parse()

	cfg, err := config.Load(cfgPath)
	if err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "unable to load config: %v", err)
		os.Exit(1)
	}

	prepareDirs(cfg)
	zlog := logger(cfg)
	wrk, err := worker.NewWorker(cfg, zlog)
	if err != nil {
		panic(err)
	}

	stopChan := make(chan os.Signal, 1)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)

	errChan := make(chan error, 1)
	go func() {
		if err := wrk.Start(); err != nil {
			errChan <- err
		}
	}()

	select {
	case <-stopChan:
		ctx, cancel := context.WithTimeout(context.Background(), wrk.ShutdownTTL())
		defer cancel()

		if err := wrk.Shutdown(ctx); err != nil {
			zlog.Error("failed to shutdown worker", log.Error(err))
		}
	case err := <-errChan:
		zlog.Error("unexpected worker error", log.Error(err))
	}
}
