package abc

import (
	"context"
	"crypto/tls"
	"fmt"
	"strconv"
	"time"

	"github.com/go-resty/resty/v2"
	"github.com/karlseguin/ccache/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

const (
	DefaultHost = "https://abc-back.yandex-team.ru"

	cacheSize = 4096
	cacheTTL  = 30 * time.Minute
)

var cache = ccache.New(ccache.Configure().MaxSize(cacheSize))

type Client struct {
	authToken string
	httpc     *resty.Client
	logger    log.Structured
}

func NewClient(authToken string, opts ...Option) (*Client, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, fmt.Errorf("failed to initialize cert pool: %w", err)
	}

	r := resty.New().
		SetTLSClientConfig(&tls.Config{RootCAs: certPool}).
		SetRetryCount(5).
		SetRedirectPolicy(resty.NoRedirectPolicy()).
		SetBaseURL(DefaultHost).
		SetHeader("Authorization", "OAuth "+authToken).
		SetHeader("User-Agent", "X-Ray <security@yandex-team.ru>")

	c := &Client{
		authToken: authToken,
		httpc:     r,
		logger:    &nop.Logger{},
	}

	for _, opt := range opts {
		opt(c)
	}

	return c, nil
}

func (c *Client) GetServiceTags(ctx context.Context, serviceID int) ([]Tag, error) {
	item, err := cache.Fetch(fmt.Sprintf("%d@tags", serviceID), cacheTTL, func() (interface{}, error) {
		var result tagsResponse
		rsp, err := c.httpc.R().
			SetContext(ctx).
			SetQueryParams(map[string]string{
				"fields": "tags",
				"id":     strconv.Itoa(serviceID),
			}).
			SetResult(&result).
			Get("/api/v4/services/")

		if err != nil {
			return nil, err
		}

		if !rsp.IsSuccess() {
			return nil, fmt.Errorf("failed to request ABC, non-200 status code: %d", rsp.StatusCode())
		}

		var tags []Tag
		for _, r := range result.Results {
			tags = append(tags, r.Tags...)
		}

		return tags, nil
	})

	if err != nil {
		return nil, err
	}

	return item.Value().([]Tag), nil
}

func IsSoxTags(tags []Tag) bool {
	for _, tag := range tags {
		if tag.ID == SoxServiceTag {
			return true
		}
	}

	return false
}

func CompactTags(tags []Tag) []string {
	out := make([]string, len(tags))
	for i, t := range tags {
		out[i] = fmt.Sprintf("%s (%d)", t.Name.En, t.ID)
	}
	return out
}
