package ant

import (
	"context"
	"crypto/tls"
	"errors"
	"fmt"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
)

const (
	APIHost     = "ant.sec.yandex-team.ru"
	APIEndpoint = "https://" + APIHost + "/api/v1/validate/"

	httpTimeout = 5 * time.Second
	httpRetries = 2
)

type (
	Client struct {
		httpc *resty.Client
		token string
	}

	SecretInfo struct {
		Ok       bool `json:"ok"`
		Valid    bool `json:"valid"`
		YisKnows bool `json:"yis_knows"`
	}

	ValidateRequest struct {
		Secret string `json:"secret"`
	}
)

func New(l log.Logger) *Client {
	httpc := resty.New().
		SetHeader("Content-Type", "application/json; charset=utf-8").
		SetTimeout(httpTimeout).
		SetRetryCount(httpRetries).
		SetLogger(l.Fmt()).
		SetBaseURL(APIEndpoint)

	certPool, err := certifi.NewCertPool()
	if err != nil {
		l.Error("ant-secret client: failed to configure TLS cert pool", log.Error(err))
	} else {
		httpc.SetTLSClientConfig(&tls.Config{RootCAs: certPool})
	}

	return &Client{
		httpc: httpc,
	}
}

func (c *Client) IsValidSecret(ctx context.Context, secret, typ string) (valid bool, err error) {
	var out SecretInfo
	handler := c.getValidatorHandler(typ)
	if handler == "" {
		return false, fmt.Errorf("unsupported secret type: %s", typ)
	}
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetResult(&out).
		SetBody(ValidateRequest{Secret: secret}).
		Post(handler)

	if err != nil {
		return out.Valid, err
	}

	if !rsp.IsSuccess() {
		return out.Valid, fmt.Errorf("wrong status code for %s/%s: %s", APIHost, handler, rsp.Status())
	}

	if !out.Ok {
		return out.Valid, errors.New("ok: false")
	}

	return out.Valid, nil
}

func (c *Client) getValidatorHandler(secretType string) (handler string) {
	switch secretType {
	case "YandexOAuth", "YOAuthInternal", "YOAuthExternal":
		return "yandex-oauth"
	case "Github", "GithubInternal", "GithubExternal":
		return "github"
	//case "IAMApiKey":
	//	return "iam-apikey"
	//case "IAMCookie":
	//	return "iam-cookie"
	//case "IAMToken":
	//	return "iam-token"
	case "JWT":
		return "jwt"
	case "Slack":
		return "slack"
	case "Telegram":
		return "telegram"
	case "TVM":
		return "tvm"
	case "Xiva":
		return "xiva"
	case "YandexSession", "YSessionInternal", "YSessionExternal":
		return "yandex-session"
	//case "S3MDS":
	//	return ""
	case "SSL", "SSLKey":
		return "ssl"
	case "StaffSSH":
		return "staff-ssh"
	// TODO(anton-k): specify correct type
	case "StaffRawSSH":
		return "staff-raw-ssh"
	default:
		return ""
	}
}
