package boxer

import (
	"fmt"

	"a.yandex-team.ru/security/libs/go/lineage"
	"a.yandex-team.ru/security/xray/internal/boxer/boxer_out"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

var (
	actualUbuntuCodenames = map[string]struct{}{
		"jammy":  {},
		"impish": {},
		"focal":  {},
		"bionic": {},
	}

	actualDebianCodenames = map[string]struct{}{
		"bookworm": {},
		"bullseye": {},
		"buster":   {},
		"stretch":  {},
	}

	actualAlpineCodenames = map[string]struct{}{
		"v3.15": {},
		"v3.14": {},
		"v3.13": {},
	}

	supportedDistros = map[string]isObsoleteFn{
		lineage.Ubuntu: isObsoleteUbuntu,
		//lineage.Debian: isObsoleteDebian,
		//lineage.Alpine: isObsoleteAlpine,
	}

	references = map[string]string{
		lineage.Ubuntu: "https://wiki.ubuntu.com/Releases",
		lineage.Debian: "https://wiki.debian.org/DebianReleases",
		lineage.Alpine: "https://alpinelinux.org/releases/",
	}
)

type isObsoleteFn func(codename string) bool

func NewBoxerInfo() (*boxer_out.Result, error) {
	currentOS, err := lineage.CurrentOS()
	if err != nil {
		return nil, fmt.Errorf("failed to detect OS: %w", err)
	}

	var kind xrayrpc.BoxerIssueDetail_Kind
	checkIsObsolete, isSupported := supportedDistros[currentOS.Family]
	switch {
	case !isSupported:
		kind = xrayrpc.BoxerIssueDetail_BIK_CONFUSED
	case checkIsObsolete(currentOS.Codename):
		kind = xrayrpc.BoxerIssueDetail_BIK_OBSOLETE
	default:
		kind = xrayrpc.BoxerIssueDetail_BIK_UNSPECIFIED
	}

	return &boxer_out.Result{
		Issue: &xrayrpc.BoxerIssueDetail{
			Kind:      kind,
			Reference: references[currentOS.Family],
			Distro: &xrayrpc.BoxerIssueDetail_Distro{
				Family:   currentOS.Family,
				Release:  currentOS.Release,
				Codename: currentOS.Codename,
			},
		},
	}, nil
}

func CollectInfo() (*boxer_out.CollectorResult, error) {
	currentOS, err := lineage.CurrentOS()
	if err != nil {
		return nil, fmt.Errorf("failed to detect OS: %w", err)
	}

	return &boxer_out.CollectorResult{
		Finding: &xrayrpc.BoxerFindingDetail{
			Distro: &xrayrpc.BoxerFindingDetail_Distro{
				Family:   currentOS.Family,
				Release:  currentOS.Release,
				Codename: currentOS.Codename,
			}},
	}, nil
}

func ProcessInfo(info *boxer_out.CollectorResult) (*boxer_out.Result, error) {
	var kind xrayrpc.BoxerIssueDetail_Kind
	checkIsObsolete, isSupported := supportedDistros[info.Finding.Distro.Family]
	switch {
	case !isSupported:
		kind = xrayrpc.BoxerIssueDetail_BIK_CONFUSED
	//(use only Codename
	case checkIsObsolete(info.Finding.Distro.Codename):
		kind = xrayrpc.BoxerIssueDetail_BIK_OBSOLETE
	default:
		kind = xrayrpc.BoxerIssueDetail_BIK_UNSPECIFIED
	}

	return &boxer_out.Result{
		Issue: &xrayrpc.BoxerIssueDetail{
			Kind:      kind,
			Reference: references[info.Finding.Distro.Family],
			Distro: &xrayrpc.BoxerIssueDetail_Distro{
				Family:   info.Finding.Distro.Family,
				Release:  info.Finding.Distro.Release,
				Codename: info.Finding.Distro.Codename,
			},
		},
	}, nil
}

func isObsoleteUbuntu(codename string) bool {
	_, ok := actualUbuntuCodenames[codename]
	return !ok
}

func isObsoleteDebian(codename string) bool {
	_, ok := actualDebianCodenames[codename]
	return !ok
}

func isObsoleteAlpine(codename string) bool {
	_, ok := actualAlpineCodenames[codename]
	return !ok
}
