package cli

import (
	"context"
	"fmt"

	"github.com/spf13/cobra"
	"google.golang.org/protobuf/encoding/prototext"

	"a.yandex-team.ru/security/xray/pkg/xray"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

var analysisCmd = &cobra.Command{
	Use: "analysis",
	RunE: func(_ *cobra.Command, args []string) error {
		if analysisFlags.ID == "" {
			return fmt.Errorf("empty analysis ID")
		}

		client, err := xray.NewClient(clientOpts()...)
		if err != nil {
			return fmt.Errorf("failed create client: %w", err)
		}
		defer closeClient(client)

		result, err := client.GetAnalysis(context.Background(), analysisFlags.ID)
		if err != nil {
			return err
		}

		return printAnalysisResult(result)
	},
}

var (
	analysisFlags struct {
		ID string
	}
)

func init() {
	flags := analysisCmd.PersistentFlags()
	flags.StringVar(&analysisFlags.ID, "id", "", "analysis ID")
	RootCmd.AddCommand(analysisCmd)
}

func printAnalysisResult(reply *xrayrpc.AnalysisGetReply) error {
	fmt.Printf("status: %s\n", reply.Status.String())
	if reply.StatusDescription != "" {
		fmt.Printf("status reason: %s\n", reply.StatusDescription)
	}
	fmt.Printf("updated at: %s\n", reply.UpdatedAt.AsTime())
	fmt.Printf("log: %s\n", reply.LogUri)
	fmt.Println("results:")
	data, err := prototext.Marshal(reply.Results)
	if err != nil {
		return fmt.Errorf("can't parshal proto response: %w", err)
	}
	fmt.Println(string(data))
	return nil
}
