package cli

import (
	"fmt"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/security/xray/pkg/xray"
)

var RootCmd = &cobra.Command{
	Use:          "xray",
	SilenceUsage: true,
	Short:        "Y.Deploy stage checker",
}

var (
	apiAddr  string
	insecure bool
	logger   = func() log.Logger {
		zlog, err := zap.New(zap.ConsoleConfig(log.InfoLevel))
		if err != nil {
			panic(fmt.Sprintf("failed to create logger: %s", err))
		}
		return zlog
	}()
)

func init() {
	flags := RootCmd.PersistentFlags()
	flags.StringVar(&apiAddr, "addr", "", "custom X-Ray API addr")
	flags.BoolVar(&insecure, "insecure", false, "make insecure connection")
}

func Execute() {
	if err := RootCmd.Execute(); err != nil {
		os.Exit(1)
	}
}

func clientOpts() []xray.Option {
	opts := []xray.Option{
		xray.WithLogger(logger.Structured()),
		xray.WithAddr(apiAddr),
		xray.WithInsecure(insecure),
	}

	if authToken := os.Getenv("XRAY_TOKEN"); authToken != "" {
		opts = append(opts, xray.WithTokenAuth(authToken))
	}
	return opts
}

func closeClient(c *xray.Client) {
	if err := c.Close(); err != nil {
		logger.Error("failed to close X-Ray client", log.Error(err))
	}
}
