package cli

import (
	"context"
	"errors"
	"fmt"
	"time"

	"github.com/spf13/cobra"
	"google.golang.org/protobuf/encoding/prototext"

	"a.yandex-team.ru/security/xray/pkg/xray"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

var stageCmd = &cobra.Command{
	Use: "stage",
}

var stageScheduleCmd = &cobra.Command{
	Use: "schedule",
	RunE: func(_ *cobra.Command, args []string) error {
		if stageScheduleFlags.Latest {
			if stageScheduleFlags.ID == "" {
				return errors.New("empty stage ID")
			}
		} else {
			if err := stageScheduleFlags.stage.Validate(); err != nil {
				return err
			}
		}

		client, err := xray.NewClient(clientOpts()...)
		if err != nil {
			return fmt.Errorf("failed create client: %w", err)
		}
		defer closeClient(client)

		var analyzeID string
		if stageScheduleFlags.Latest {
			result, err := client.ScheduleLatest(
				context.Background(),
				&xrayrpc.StageScheduleLatestRequest{
					StageId:     stageScheduleFlags.ID,
					Description: stageScheduleFlags.Description,
					Force:       stageScheduleFlags.Force,
				},
			)
			if err != nil {
				return err
			}

			fmt.Println("stage analysis scheduled")
			fmt.Printf("  - stage id: %s\n", result.GetStageId())
			fmt.Printf("  - stage uuid: %s\n", result.GetStageUuid())
			fmt.Printf("  - stage revision: %d\n", result.GetStageRevision())
			fmt.Printf("  - analyze id: %s\n", result.GetAnalyzeId())
			analyzeID = result.GetAnalyzeId()
		} else {
			result, err := client.Schedule(
				context.Background(),
				&xrayrpc.StageScheduleRequest{
					Stage: &xrayrpc.Stage{
						Id:       stageScheduleFlags.stage.ID,
						Uuid:     stageScheduleFlags.stage.UUID,
						Revision: stageScheduleFlags.stage.Revision,
					},
					Description: stageScheduleFlags.Description,
					Force:       stageScheduleFlags.Force,
				},
			)
			if err != nil {
				return err
			}

			fmt.Printf("stage analysis scheduled: %s\n", result.GetAnalyzeId())
			analyzeID = result.GetAnalyzeId()
		}

		if !stageScheduleFlags.Wait {
			return nil
		}

		ticker := time.NewTicker(2 * time.Second)
		for {
			<-ticker.C

			result, err := client.GetAnalysis(context.Background(), analyzeID)
			if err != nil {
				fmt.Printf("failed to get results: %s\n", err)
				continue
			}

			ticker.Stop()
			err = printAnalysisResult(result)
			if err != nil {
				fmt.Printf("failed to print results: %s\n", err)
				continue
			}
			break
		}

		return nil
	},
}

var stageResultsCmd = &cobra.Command{
	Use: "results",
	RunE: func(_ *cobra.Command, args []string) error {
		if err := stageResultsFlags.stage.Validate(); err != nil {
			return err
		}

		client, err := xray.NewClient(clientOpts()...)
		if err != nil {
			return fmt.Errorf("failed create client: %w", err)
		}
		defer closeClient(client)

		result, err := client.GetResults(context.Background(), &xrayrpc.Stage{
			Id:       stageResultsFlags.stage.ID,
			Uuid:     stageResultsFlags.stage.UUID,
			Revision: stageResultsFlags.stage.Revision,
		})
		if err != nil {
			return err
		}

		fmt.Printf("status: %s\n", result.AnalysisStatus.String())
		if result.StatusDescription != "" {
			fmt.Printf("status reason: %s\n", result.StatusDescription)
		}
		fmt.Printf("updated at: %s\n", result.UpdatedAt.AsTime())
		fmt.Printf("log: %s\n", result.LogUri)
		fmt.Println("results:")
		data, err := prototext.Marshal(result.Results)
		if err != nil {
			return fmt.Errorf("failed to marshal proto: %w", err)
		}
		fmt.Println(string(data))
		return nil
	},
}

var listLatestCmd = &cobra.Command{
	Use: "list",
	RunE: func(_ *cobra.Command, args []string) error {
		client, err := xray.NewClient(clientOpts()...)
		if err != nil {
			return fmt.Errorf("failed create client: %w", err)
		}
		defer closeClient(client)

		result, err := client.List(context.Background(), &xrayrpc.StageListRequest{
			Limit: 50,
		})
		if err != nil {
			return err
		}

		for i, stageStatus := range result.Statuses {
			fmt.Printf("  - [%d] %s:\n", i+1, stageStatus.Stage.Id)
			fmt.Printf("    revision: %d\n", stageStatus.Stage.Revision)
			fmt.Printf("    updated at: %s\n", stageStatus.UpdatedAt.AsTime().Format(time.RFC3339))
			fmt.Printf("    status: %s\n", stageStatus.AnalysisStatus)
			fmt.Printf("    overview: low:%d medium:%d high:%d\n",
				stageStatus.Issues.Low, stageStatus.Issues.Medium, stageStatus.Issues.High)
			fmt.Println()
		}
		return nil
	},
}

type (
	stage struct {
		ID       string
		UUID     string
		Revision uint32
	}
)

func (s *stage) Validate() error {
	switch {
	case s.Revision == 0:
		return errors.New("empty stage Revision")
	case s.ID == "":
		return errors.New("empty stage ID")
	case s.UUID == "":
		return errors.New("empty stage UUID")
	default:
		return nil
	}
}

var (
	stageScheduleFlags struct {
		stage
		Description string
		Force       bool
		Latest      bool
		Wait        bool
	}

	stageResultsFlags struct {
		stage
	}
)

func init() {
	{
		flags := stageScheduleCmd.PersistentFlags()
		flags.StringVar(&stageScheduleFlags.stage.ID, "id", "", "Y.Deploy stage ID")
		flags.StringVar(&stageScheduleFlags.stage.UUID, "uuid", "", "Y.Deploy stage UUID")
		flags.Uint32Var(&stageScheduleFlags.stage.Revision, "revision", 0, "Y.Deploy stage Revision")
		flags.BoolVar(&stageScheduleFlags.Latest, "latest", false, "schedule latest stage version (ignore uuid/revision)")
		flags.BoolVar(&stageScheduleFlags.Force, "force", false, "force scheduling")
		flags.BoolVar(&stageScheduleFlags.Wait, "wait", false, "wait results")
		flags.StringVar(&stageScheduleFlags.Description, "description", "", "Shedule description")
	}

	{
		flags := stageResultsCmd.PersistentFlags()
		flags.StringVar(&stageResultsFlags.stage.ID, "id", "", "Y.Deploy stage ID")
		flags.StringVar(&stageResultsFlags.stage.UUID, "uuid", "", "Y.Deploy stage UUID")
		flags.Uint32Var(&stageResultsFlags.stage.Revision, "revision", 0, "Y.Deploy stage Revision")
	}

	stageCmd.AddCommand(stageScheduleCmd)
	stageCmd.AddCommand(stageResultsCmd)
	stageCmd.AddCommand(listLatestCmd)
	RootCmd.AddCommand(stageCmd)
}
