package db

import (
	"fmt"
)

func selectAnalysisQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $id AS Utf8;

SELECT
	id, created_at, updated_at, result_path, log_path, status, status_description
FROM
	analyzes
WHERE
	key = Digest::CityHash($id) AND id = $id
LIMIT 1;`, pathPrefix)
}

func selectStageAnalysisQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $stageUUID AS Utf8;
DECLARE $stageRevision AS Uint32;

$stage = (
SELECT
	analyze_id, overview
FROM
	stages
WHERE
	key = Digest::CityHash($stageUUID || CAST($stageRevision AS String))
	AND uuid = $stageUUID
	AND revision = $stageRevision
LIMIT 1
);

SELECT
	analyze.id,
	analyze.updated_at,
	analyze.result_path,
	analyze.log_path,
	analyze.status,
	analyze.status_description,
	stage.overview as overview
FROM $stage as stage
INNER JOIN analyzes as analyze ON (analyze.key == Digest::CityHash(stage.analyze_id) AND analyze.id == stage.analyze_id)
LIMIT 1;
`, pathPrefix)
}

func selectStageStatusQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $uuid AS Utf8;
DECLARE $revision AS Uint32;

SELECT
	uuid, revision, id, updated_at, overview, status
FROM
	stages
WHERE
	key = Digest::CityHash($uuid || CAST($revision AS String)) AND uuid = $uuid AND revision = $revision
LIMIT 1;`, pathPrefix)
}

func selectAnalysisStatusQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $id AS Utf8;

SELECT
	status, status_description
FROM
	analyzes
WHERE
	key = Digest::CityHash($id) AND id = $id
LIMIT 1;`, pathPrefix)
}

func selectAnalysisLogQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $id AS Utf8;

SELECT
	log_path
FROM
	analyzes
WHERE
	key = Digest::CityHash($id) AND id = $id
LIMIT 1;`, pathPrefix)
}

func selectLatestStageQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $uuid AS Utf8;

SELECT
	revision
FROM
	latest_stages
WHERE
	key = Digest::CityHash($uuid) AND uuid = $uuid
LIMIT 1;`, pathPrefix)
}

func selectStagesStatusQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $limit AS Int32;

$latest_stages = (
SELECT
    Digest::CityHash(stage_uuid || CAST(stage_revision AS String)) as key, created_at, stage_id, stage_uuid, stage_revision, analyze_id
FROM latest_stage_analyzes
ORDER BY created_at DESC
LIMIT $limit
);

SELECT
    ls.created_at, s.updated_at, ls.stage_id, ls.stage_uuid, ls.stage_revision, s.overview, s.status
FROM $latest_stages AS ls
INNER JOIN stages AS s
ON
    s.key = ls.key
    AND s.uuid = ls.stage_uuid
    AND s.revision = ls.stage_revision
;`, pathPrefix)
}

func selectLatestStagesStatusQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $limit AS Int32;
DECLARE $uuids AS "List<Utf8>";

$latest_stages = (
	SELECT
		ls.uuid as uuid, ls.revision as revision
	FROM AS_TABLE(ListMap($uuids, ($uuid) -> { RETURN AsStruct(Digest::CityHash($uuid) AS key, $uuid AS uuid); })) AS uuids
	INNER JOIN latest_stages
	AS ls
	ON ls.key = uuids.key AND ls.uuid = uuids.uuid
	LIMIT $limit
);

SELECT
    s.updated_at as updated_at,
    s.id as stage_id,
    ls.uuid as stage_uuid,
    ls.revision as stage_revision,
    s.overview as overview,
    s.status as status
FROM $latest_stages AS ls
INNER JOIN stages AS s
ON
    s.key = Digest::CityHash(ls.uuid || CAST(ls.revision AS String))
    AND s.uuid = ls.uuid
    AND s.revision = ls.revision
ORDER BY updated_at DESC
LIMIT
    $limit
;`, pathPrefix)
}

func scheduleAnalyzeQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $analyzeID AS Utf8;
DECLARE $stageUUID AS Utf8;
DECLARE $stageRevision AS Uint32;
DECLARE $latestStageRevision AS Uint32;
DECLARE $stageID AS Utf8;
DECLARE $description AS Utf8;
DECLARE $status AS Int32;
DECLARE $createdAt AS Int64;

REPLACE INTO analyzes
	(key, id, created_at, updated_at, status)
VALUES
	(Digest::CityHash($analyzeID), $analyzeID, $createdAt, $createdAt, $status)
;

REPLACE INTO stages
	(key, uuid, revision, id, description, updated_at, analyze_id, status)
VALUES
	(Digest::CityHash($stageUUID || CAST($stageRevision AS String)), $stageUUID, $stageRevision, $stageID, $description, $createdAt, $analyzeID, $status)
;

REPLACE INTO latest_stages
	(key, uuid, revision)
VALUES
	(Digest::CityHash($stageUUID), $stageUUID, $latestStageRevision)
;

REPLACE INTO latest_stage_analyzes
	(created_at, stage_id, stage_uuid, stage_revision, analyze_id)
VALUES
	($createdAt, $stageID, $stageUUID, $stageRevision, $analyzeID)
;`, pathPrefix)
}

func completeAnalyzeQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $analyzeID AS Utf8;
DECLARE $stageUUID AS Utf8;
DECLARE $stageRevision AS Uint32;
DECLARE $updatedAt AS Int64;
DECLARE $resultPath AS Utf8;
DECLARE $logPath AS Utf8;
DECLARE $status AS Int32;
DECLARE $statusDescription AS Utf8;
DECLARE $overview AS String;

UPSERT INTO	analyzes
	(key, id, updated_at, result_path, log_path, status, status_description)
VALUES
	(Digest::CityHash($analyzeID), $analyzeID, $updatedAt, $resultPath, $logPath, $status, $statusDescription)
;

UPSERT INTO stages
	(key, uuid, revision, updated_at, analyze_id, overview, status)
VALUES
	(Digest::CityHash($stageUUID || CAST($stageRevision AS String)), $stageUUID, $stageRevision, $updatedAt, $analyzeID, $overview, $status)
;`, pathPrefix)
}
