package dockinfo

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strings"

	"a.yandex-team.ru/security/libs/go/yahttp"
	"a.yandex-team.ru/security/xray/internal/httputils"
)

const (
	resolverURI = "https://dockinfo.yandex-team.ru/api/docker/resolve?"
	defaultTag  = "latest"
)

type (
	ImageInfo struct {
		Digest string  `json:"digest"`
		Layers []Layer `json:"layers"`
	}

	Layer struct {
		Digest      string `json:"digest"`
		URI         string `json:"uri"`
		Description string `json:"desc,omitempty"`
	}

	metadata struct {
		Hash   string `json:"hash"`
		Layers []struct {
			Digest string `json:"digest"`
			URI    string `json:"http"`
		} `json:"layers2"`
	}
)

func Resolve(ctx context.Context, uri string) (*ImageInfo, error) {
	var imageURI, tag string
	column := strings.LastIndex(uri, ":")
	if column == -1 {
		imageURI = uri
		tag = defaultTag
	} else {
		imageURI = uri[0:column]
		tag = uri[column+1:]
	}

	uri = resolverURI + url.Values{
		"registryUrl": {imageURI},
		"tag":         {tag},
	}.Encode()

	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		return nil, fmt.Errorf("failed to create image request: %w", err)
	}

	req = req.WithContext(ctx)
	req.Header.Set("User-Agent", "X-Ray <security@yandex-team.ru>")
	resp, err := httputils.HTTPClient.Do(req)
	if err != nil {
		return nil, fmt.Errorf("failed to get image info: %w", err)
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("failed to get image info: invalid status code %d", resp.StatusCode)
	}

	var data metadata
	err = json.NewDecoder(resp.Body).Decode(&data)
	if err != nil {
		return nil, fmt.Errorf("failed to decode image meta: %w", err)
	}

	result := &ImageInfo{
		Digest: data.Hash,
		Layers: make([]Layer, len(data.Layers)),
	}

	for i, l := range data.Layers {
		result.Layers[i] = Layer{
			Digest: l.Digest,
			URI:    l.URI,
		}
	}
	return result, nil
}
