package procutils

import (
	"bytes"
	"errors"
	"fmt"
	"os/exec"
	"syscall"
)

type CmdResult struct {
	Stdout   []byte
	Stderr   []byte
	ExitCode int
}

func RunCmd(cmd *exec.Cmd) (*CmdResult, error) {
	if cmd.Stdout != nil {
		return nil, errors.New("stdout already set")
	}
	if cmd.Stderr != nil {
		return nil, errors.New("stderr already set")
	}

	var stdout bytes.Buffer
	cmd.Stdout = &stdout
	var stderr bytes.Buffer
	cmd.Stderr = &stderr

	exitCode := 0
	if err := cmd.Run(); err != nil {
		if exitErr, ok := err.(*exec.ExitError); ok {
			// The program has exited with an exit code != 0
			if status, ok := exitErr.Sys().(syscall.WaitStatus); ok {
				exitCode = status.ExitStatus()
			}
		} else {
			return nil, fmt.Errorf("wait error \"%s\"", err.Error())
		}
	}

	return &CmdResult{
		Stdout:   stdout.Bytes(),
		Stderr:   stderr.Bytes(),
		ExitCode: exitCode,
	}, nil
}
