package queue

import (
	"context"
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"

	"a.yandex-team.ru/library/go/ptr"
)

const (
	SqsRegion                = "yandex"
	MaxMessageSize           = 250 * 1000
	DefaultMaxMsg      int64 = 1
	DefaultPollSeconds int64 = 20
	DefaultRetries           = 3
)

type (
	AuthType int

	Queue struct {
		sqs *sqs.SQS
	}

	SendOptions struct {
		// SQS queue
		QueueURL string

		// Message to send
		Msg string
	}

	ReceiveOptions struct {
		// SQS queue
		QueueURL string

		// The maximum number of messages to return. SQS never returns more messages
		// than this value (however, fewer messages might be returned).
		// Valid values: 1 to 10
		// Default: 1
		MaxNumberOfMessages int64

		// The duration (in seconds) for which the call waits for a message to arrive
		// in the queue before returning. If a message is available, the call returns
		// sooner than WaitTimeSeconds. If no messages are available and the wait time
		// expires, the call returns successfully with an empty list of messages.
		WaitTimeSeconds int64
	}

	DeleteOptions struct {
		// SQS queue
		QueueURL string

		// The receipt handle associated with the message to delete.
		ReceiptHandle *string
	}

	ChangeMessageVisibilityOptions struct {
		// SQS queue
		QueueURL string

		// The receipt handle associated with the message whose visibility timeout is
		// changed. This parameter is returned by the ReceiveMessage action.
		ReceiptHandle *string

		// The new value for the message's visibility timeout (in seconds).
		// Values values: 0 to 43200. Maximum: 12 hours.
		VisibilityTimeout int64
	}
)

func New(endpoint string, opts ...Option) (*Queue, error) {
	cfg := &aws.Config{
		Region:     ptr.String(SqsRegion),
		Endpoint:   ptr.String(endpoint),
		MaxRetries: ptr.Int(DefaultRetries),
	}

	for _, opt := range opts {
		if err := opt(cfg); err != nil {
			return nil, fmt.Errorf("failed to configure SQS session: %w", err)
		}
	}

	sess, err := session.NewSession(cfg)
	if err != nil {
		return nil, fmt.Errorf("failed to create SQS session: %w", err)
	}

	return &Queue{
		sqs: sqs.New(sess),
	}, nil
}

func (q *Queue) SendMessage(ctx context.Context, opts *SendOptions) (msgID string, resultErr error) {
	if len(opts.Msg) > MaxMessageSize {
		resultErr = fmt.Errorf("maximum message size exceed: %d > %d", len(opts.Msg), MaxMessageSize)
		return
	}

	result, err := q.sqs.SendMessageWithContext(ctx, &sqs.SendMessageInput{
		MessageBody: &opts.Msg,
		QueueUrl:    &opts.QueueURL,
	})

	if err != nil {
		resultErr = fmt.Errorf("failed to send message: %w", err)
		return
	}

	msgID = *result.MessageId
	return
}

func (q *Queue) ReceiveMessage(ctx context.Context, opts *ReceiveOptions) (msgs []*sqs.Message, resultErr error) {
	maxMsg := DefaultMaxMsg
	if opts.MaxNumberOfMessages > 0 {
		maxMsg = opts.MaxNumberOfMessages
	}

	pollWaitSeconds := DefaultPollSeconds
	if opts.WaitTimeSeconds > 0 {
		pollWaitSeconds = opts.WaitTimeSeconds
	}

	result, err := q.sqs.ReceiveMessageWithContext(
		ctx,
		&sqs.ReceiveMessageInput{
			QueueUrl:            &opts.QueueURL,
			MaxNumberOfMessages: &maxMsg,
			WaitTimeSeconds:     &pollWaitSeconds,
		},
	)

	if err != nil {
		resultErr = fmt.Errorf("failed to fetch sqs message: %w", err)
		return
	}

	msgs = result.Messages
	return
}

func (q *Queue) DeleteMessage(ctx context.Context, opts *DeleteOptions) error {
	_, err := q.sqs.DeleteMessageWithContext(ctx, &sqs.DeleteMessageInput{
		QueueUrl:      &opts.QueueURL,
		ReceiptHandle: opts.ReceiptHandle,
	})

	if err != nil {
		return fmt.Errorf("failed to delete sqs message: %w", err)
	}
	return nil
}

func (q *Queue) ChangeMessageVisibility(ctx context.Context, opts *ChangeMessageVisibilityOptions) error {
	_, err := q.sqs.ChangeMessageVisibilityWithContext(ctx, &sqs.ChangeMessageVisibilityInput{
		ReceiptHandle:     opts.ReceiptHandle,
		QueueUrl:          &opts.QueueURL,
		VisibilityTimeout: &opts.VisibilityTimeout,
	})

	if err != nil {
		return fmt.Errorf("failed to delete sqs message: %w", err)
	}
	return nil
}
