package sandbox

import (
	"context"
	"crypto/tls"
	"fmt"
	"time"

	"github.com/go-resty/resty/v2"
	"github.com/karlseguin/ccache/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
)

const (
	ProxyHost   = "proxy.sandbox.yandex-team.ru"
	ProxyURI    = "http://" + ProxyHost
	APIHost     = "sandbox.yandex-team.ru"
	APIEndpoint = "https://" + APIHost + "/api/v1.0/"

	httpTimeout = 5 * time.Second
	httpRetries = 2
	cacheTTL    = 10 * time.Hour
	cacheSize   = 2048
)

type (
	Client struct {
		httpc *resty.Client
		cache *ccache.Cache
	}

	ResourceInfo struct {
		Filename string `json:"file_name"`
		Owner    string `json:"owner"`
		Size     uint64 `json:"size"`
		State    string `json:"state"` // TODO(buglloc): enum needed
		SkynetID string `json:"skynet_id"`
	}
)

func New(l log.Logger) *Client {
	httpc := resty.New().
		SetHeader("Content-Type", "application/json; charset=utf-8").
		SetTimeout(httpTimeout).
		SetRetryCount(httpRetries).
		SetLogger(l.Fmt()).
		SetBaseURL(APIEndpoint)

	certPool, err := certifi.NewCertPool()
	if err != nil {
		l.Error("sandbox: failed to configure TLS cert pool", log.Error(err))
	} else {
		httpc.SetTLSClientConfig(&tls.Config{RootCAs: certPool})
	}

	return &Client{
		httpc: httpc,
		cache: ccache.New(
			ccache.Configure().
				MaxSize(cacheSize),
		),
	}
}

func (c *Client) GetResourceInfo(ctx context.Context, resourceID string) (ResourceInfo, error) {
	item, err := c.cache.Fetch("res:"+resourceID, cacheTTL, func() (interface{}, error) {
		var out ResourceInfo
		rsp, err := c.httpc.R().
			SetContext(ctx).
			SetResult(&out).
			SetPathParams(map[string]string{
				"resourceID": resourceID,
			}).
			Get("resource/{resourceID}")

		if err != nil {
			return out, err
		}

		if !rsp.IsSuccess() {
			return out, fmt.Errorf("wrong status code for /resource/%s: %s", resourceID, rsp.Status())
		}

		return out, nil
	})

	if err != nil {
		return ResourceInfo{}, err
	}

	return item.Value().(ResourceInfo), nil
}

func (c *Client) Close() {
	c.cache.Stop()
}
