package config

import (
	"context"
	"fmt"
	"os"
	"strings"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/xray/internal/db"
	"a.yandex-team.ru/security/xray/internal/logutil"
	"a.yandex-team.ru/security/xray/internal/storage/s3storage"
)

type SQS struct {
	Endpoint string `yaml:"endpoint"`
	Account  string `yaml:"account"`
	Queue    string `yaml:"queue"`
}

type S3 struct {
	Endpoint    string       `yaml:"endpoint"`
	AccessKeyID string       `yaml:"access_key_id"`
	Bucket      string       `yaml:"bucket"`
	TvmID       tvm.ClientID `yaml:"tvm_id"`
}

type YDB struct {
	Endpoint string `yaml:"endpoint"`
	Database string `yaml:"database"`
	Path     string `yaml:"path"`
}

type Server struct {
	Addr     string `yaml:"addr"`
	Insecure bool   `yaml:"insecure"`
	CertPath string `yaml:"cert_path"`
}

type Roles struct {
	Admin  string `yaml:"admin"`
	Reader string `yaml:"reader"`
}

type TVM struct {
	ClientID     tvm.ClientID            `yaml:"tvm_id"`
	ClientSecret string                  `yaml:"-"`
	CacheDir     string                  `yaml:"cache_dir"`
	Destinations map[string]tvm.ClientID `yaml:"dests"`
	Env          tvm.BlackboxEnv         `yaml:"env"`
	IDMSlug      string                  `yaml:"idm_slug"`
	Port         int                     `yaml:"port"`
}

type Config struct {
	LogLvl            logutil.Level  `yaml:"log_level"`
	Server            Server         `yaml:"server"`
	SQS               SQS            `yaml:"sqs"`
	S3                S3             `yaml:"s3"`
	YDB               YDB            `yaml:"ydb"`
	YpToken           string         `yaml:"yp_token"`
	SignKey           string         `yaml:"sign_key"`
	AllowedTVMClients []tvm.ClientID `yaml:"allowed_tvm_clients"`
	Roles             Roles          `yaml:"roles"`
	TVM               TVM            `yaml:"tvm"`
}

func Load(cfgPath string) (*Config, error) {
	cfg := &Config{
		LogLvl: logutil.Level{
			Level: log.DebugLevel,
		},
		YpToken: os.Getenv("YP_TOKEN"),
		SignKey: os.Getenv("SIGN_KEY"),
		TVM: TVM{
			ClientSecret: os.Getenv("TVM_SECRET_ANY"),
		},
	}

	loader := confita.NewLoader(env.NewBackend(), file.NewBackend(cfgPath))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, err
	}

	return cfg, nil
}

func (c *Config) DBConfig() *db.Options {
	return &db.Options{
		Endpoint: c.YDB.Endpoint,
		Database: c.YDB.Database,
		Path:     c.YDB.Path,
	}
}

func (c *Config) S3StorageConfig() *s3storage.Options {
	return &s3storage.Options{
		Endpoint:    c.S3.Endpoint,
		Bucket:      c.S3.Bucket,
		AccessKeyID: c.S3.AccessKeyID,
		TvmID:       c.S3.TvmID,
	}
}

func (c *Config) RequestsQueueURL() string {
	return fmt.Sprintf("%s/%s/%s",
		strings.TrimRight(c.SQS.Account, "/"),
		c.SQS.Account,
		c.SQS.Queue,
	)
}
