package infra

import (
	"context"
	"fmt"
	"os"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/security/libs/go/xtvm"
	"a.yandex-team.ru/security/xray/internal/db"
	"a.yandex-team.ru/security/xray/internal/queue"
	"a.yandex-team.ru/security/xray/internal/servers/grpc/config"
	"a.yandex-team.ru/security/xray/internal/storage/s3storage"
	"a.yandex-team.ru/yp/go/yp"
)

const tvmWaitTimeout = 30 * time.Second

type Infra struct {
	Config    *config.Config
	DB        *db.DB
	Queue     *queue.Queue
	TVM       tvm.Client
	S3Storage *s3storage.Storage
	YP        *yp.Client
	BlackBox  blackbox.Client
	Logger    log.Logger
}

func New(cfg *config.Config) *Infra {
	zlog, err := zap.NewDeployLogger(cfg.LogLvl.Level)
	if err != nil {
		panic(fmt.Sprintf("failed to create logger: %s", err))
	}

	return &Infra{
		Config: cfg,
		Logger: zlog,
	}
}

func (c *Infra) Initialize() (err error) {
	c.TVM, err = c.newTVMClient()
	if err != nil {
		return fmt.Errorf("create TVM client: %w", err)
	}

	if err := xtvm.Wait(c.TVM, tvmWaitTimeout); err != nil {
		return fmt.Errorf("wait tvm: %w", err)
	}

	c.Queue, err = queue.New(
		c.Config.SQS.Endpoint,
		queue.WithAuthTVM(c.Config.SQS.Account, c.TVM),
	)
	if err != nil {
		return fmt.Errorf("create SQS client: %w", err)
	}

	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	c.DB, err = db.New(ctx, c.TVM, c.Config.DBConfig())
	if err != nil {
		return fmt.Errorf("create YDB client: %w", err)
	}

	c.S3Storage, err = s3storage.NewS3Storage(c.TVM, c.Config.S3StorageConfig())
	if err != nil {
		return fmt.Errorf("create S3 client: %w", err)
	}

	c.BlackBox, err = httpbb.NewIntranet(
		httpbb.WithTVM(c.TVM),
		httpbb.WithLogger(c.Logger.Structured()),
	)
	if err != nil {
		return fmt.Errorf("create Blackbox client: %w", err)
	}

	c.YP, err = yp.NewClient("xdc", yp.WithLogger(c.Logger), yp.WithAuthToken(c.Config.YpToken))
	if err != nil {
		return fmt.Errorf("create YP client: %w", err)
	}

	return
}

func (c *Infra) Finish(ctx context.Context) (err error) {
	if c.DB != nil {
		err = c.DB.Reset(ctx)
		if err != nil {
			return
		}
	}
	return
}

func (c *Infra) newTVMClient() (tvm.Client, error) {
	if c.Config.TVM.CacheDir != "" {
		if err := os.MkdirAll(c.Config.TVM.CacheDir, 0o700); err != nil {
			return nil, fmt.Errorf("unable to create tvm cache dir: %w", err)
		}
	}

	tvmSettings := tvmauth.TvmAPISettings{
		SelfID:                      c.Config.TVM.ClientID,
		ServiceTicketOptions:        tvmauth.NewAliasesOptions(c.Config.TVM.ClientSecret, c.Config.TVM.Destinations),
		DiskCacheDir:                c.Config.TVM.CacheDir,
		BlackboxEnv:                 &c.Config.TVM.Env,
		FetchRolesForIdmSystemSlug:  c.Config.TVM.IDMSlug,
		DisableDefaultUIDCheck:      true,
		DisableSrcCheck:             true,
		EnableServiceTicketChecking: true,
	}

	if c.Config.TVM.Port != 0 {
		tvmSettings.TVMPort = c.Config.TVM.Port
	}

	return tvmauth.NewAPIClient(tvmSettings, c.Logger)
}
