package grpc

import (
	"context"
	"fmt"
	"net"

	grpcMiddleware "github.com/grpc-ecosystem/go-grpc-middleware"
	grpcAuth "github.com/grpc-ecosystem/go-grpc-middleware/auth"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/servers/grpc/auth"
	"a.yandex-team.ru/security/xray/internal/servers/grpc/infra"
	"a.yandex-team.ru/security/xray/internal/servers/grpc/services"
	"a.yandex-team.ru/security/xray/pkg/xrayerrors"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

type (
	Server struct {
		i    *infra.Infra
		grpc *grpc.Server
	}
)

func New(infra *infra.Infra) (*Server, error) {
	s := &Server{
		i: infra,
	}

	opts := []grpc.ServerOption{
		grpcMiddleware.WithUnaryServerChain(
			s.errHandler,
			grpcAuth.UnaryServerInterceptor(auth.NewGrpcAuth(infra)),
		),
	}

	if !infra.Config.Server.Insecure {
		cred, err := credentials.NewServerTLSFromFile(infra.Config.Server.CertPath, infra.Config.Server.CertPath)
		if err != nil {
			return nil, fmt.Errorf("failed to create srv credentials: %w", err)
		}
		opts = append(opts, grpc.Creds(cred))
	}

	s.grpc = grpc.NewServer(opts...)
	return s, nil
}

func (s *Server) onStart() (err error) {
	return s.i.Initialize()
}

func (s *Server) onEnd() (err error) {
	return s.i.Finish(context.Background())
}

func (s *Server) ListenAndServe() error {
	err := s.onStart()
	if err != nil {
		return fmt.Errorf("failed to start server: %w", err)
	}

	defer func() {
		err := s.onEnd()
		if err != nil {
			s.i.Logger.Error("failed to stop", log.Error(err))
		}
	}()

	listener, err := net.Listen("tcp", s.i.Config.Server.Addr)
	if err != nil {
		return fmt.Errorf("failed to listen: %w", err)
	}

	defer func() {
		_ = listener.Close()
	}()

	xrayrpc.RegisterCommonServiceServer(s.grpc, &services.CommonService{
		Infra: s.i,
	})

	xrayrpc.RegisterStageServiceServer(s.grpc, &services.StageService{
		Infra: s.i,
	})

	xrayrpc.RegisterAnalysisServiceServer(s.grpc, &services.AnalysisService{
		Infra: s.i,
	})

	s.i.Logger.Info("Starting to serve gRPC", log.String("addr", listener.Addr().String()))
	return s.grpc.Serve(listener)
}

func (s *Server) GracefulStop() {
	s.grpc.GracefulStop()
}

func (s *Server) errHandler(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo, handler grpc.UnaryHandler) (interface{}, error) {
	resp, err := handler(ctx, req)
	if err != nil {
		if grpcErr, ok := status.FromError(err); !ok || grpcErr.Code() != xrayerrors.ErrCodeConflictSchedule {
			// ignore ErrCodeConflictSchedule err
			s.i.Logger.Error("request error", log.String("method", info.FullMethod), log.Error(err))
		}
	}

	return resp, err
}
