package services

import (
	"context"
	"crypto/hmac"
	"crypto/sha1"
	"encoding/base64"
	"errors"
	"fmt"
	"strings"

	grpcAuth "github.com/grpc-ecosystem/go-grpc-middleware/auth"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/db"
	"a.yandex-team.ru/security/xray/internal/servers/grpc/infra"
	"a.yandex-team.ru/security/xray/pkg/xrayerrors"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

var (
	_ xrayrpc.AnalysisServiceServer    = (*AnalysisService)(nil)
	_ grpcAuth.ServiceAuthFuncOverride = (*AnalysisService)(nil)
)

type AnalysisService struct {
	*infra.Infra
}

// No auth required
func (s *AnalysisService) AuthFuncOverride(
	ctx context.Context,
	_ string,
) (context.Context, error) {

	return ctx, nil
}

func (s *AnalysisService) Get(
	ctx context.Context,
	req *xrayrpc.AnalysisGetRequest,
) (*xrayrpc.AnalysisGetReply, error) {

	id, err := s.parseAnalyzeID(req.Id)
	if err != nil {
		return nil, fmt.Errorf("failed to parse analyzeID: %w", err)
	}

	dbAnalysis, err := s.DB.LookupAnalysis(ctx, id)
	if err != nil {
		if err == db.ErrNotFound {
			return nil, status.Error(codes.NotFound, "analysis not found")
		}

		return nil, fmt.Errorf("failed to lookup analysis: %w", err)
	}

	switch dbAnalysis.Status {
	case xrayrpc.AnalysisStatusKind_ASK_UNSPECIFIED:
		return nil, status.Error(xrayerrors.ErrCodeInvalidStageStatus, "analysis in 'UNSPECIFIED' state, something really shit happens")
	case xrayrpc.AnalysisStatusKind_ASK_START:
		return nil, status.Error(xrayerrors.ErrCodeInvalidStageStatus, "analysis is not completed yet")
	default:
	}

	result := &xrayrpc.AnalysisGetReply{
		UpdatedAt:         timestamppb.New(dbAnalysis.UpdatedAt),
		Results:           new(xrayrpc.AnalyzeResult),
		Status:            dbAnalysis.Status,
		StatusDescription: dbAnalysis.StatusDescription,
	}

	if dbAnalysis.LogPath != "" {
		result.LogUri = s.S3Storage.FileURI(dbAnalysis.LogPath)
	}

	if dbAnalysis.ResultPath != "" {
		rawResult, err := s.S3Storage.DownloadFile(dbAnalysis.ResultPath)
		if err != nil {
			s.Logger.Error("failed to download results", log.Error(err), log.String("path", dbAnalysis.ResultPath))
		} else {
			if err = proto.Unmarshal(rawResult, result.Results); err != nil {
				s.Logger.Error("failed to parse results", log.Error(err), log.String("path", dbAnalysis.ResultPath))
			}
		}
	}

	return result, nil
}

func (s *AnalysisService) GetStatus(
	ctx context.Context,
	req *xrayrpc.AnalysisGetStatusRequest,
) (*xrayrpc.AnalysisGetStatusReply, error) {

	id, err := s.parseAnalyzeID(req.Id)
	if err != nil {
		return nil, fmt.Errorf("failed to parse analyzeID: %w", err)
	}

	dbAnalysis, err := s.DB.LookupAnalysisStatus(ctx, id)
	if err != nil {
		return nil, fmt.Errorf("failed to lookup analysis: %w", err)
	}

	return &xrayrpc.AnalysisGetStatusReply{
		Status:      dbAnalysis.Value,
		Description: dbAnalysis.Description,
	}, nil
}

func (s *AnalysisService) parseAnalyzeID(data string) (string, error) {
	parts := strings.SplitN(data, ":", 2)
	if len(parts) != 2 {
		return "", errors.New("invalid format")
	}

	actualMAC, err := base64.StdEncoding.DecodeString(parts[1])
	if err != nil {
		return "", fmt.Errorf("failed to parse sign: %w", err)
	}

	mac := hmac.New(sha1.New, []byte(s.Config.SignKey))
	_, _ = mac.Write([]byte(parts[0]))
	expectedMAC := mac.Sum(nil)

	if !hmac.Equal(actualMAC, expectedMAC) {
		return "", errors.New("sign is invalid")
	}

	return parts[0], nil
}
