package config

import (
	"context"
	"os"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/xray/internal/logutil"
)

type Xray struct {
	Addr     string       `yaml:"addr"`
	Insecure bool         `yaml:"insecure"`
	TvmID    tvm.ClientID `yaml:"tvm_id"`
}

type TVM struct {
	ClientID     tvm.ClientID            `yaml:"tvm_id"`
	ClientSecret string                  `yaml:"-"`
	CacheDir     string                  `yaml:"cache_dir"`
	Destinations map[string]tvm.ClientID `yaml:"dests"`
	Env          tvm.BlackboxEnv         `yaml:"env"`
	IDMSlug      string                  `yaml:"idm_slug"`
	Port         int                     `yaml:"port"`
}

type Config struct {
	LogLvl logutil.Level `yaml:"log_level"`
	Addr   string        `yaml:"addr"`
	Xray   Xray          `yaml:"xray"`
	TVM    TVM           `yaml:"tvm"`
}

func Load(cfgPath string) (*Config, error) {
	cfg := &Config{
		LogLvl: logutil.Level{
			Level: log.DebugLevel,
		},
		TVM: TVM{
			ClientSecret: os.Getenv("TVM_SECRET_HUMANIZER"),
		},
	}

	loader := confita.NewLoader(env.NewBackend(), file.NewBackend(cfgPath))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, err
	}

	return cfg, nil
}
