package api

import (
	"fmt"
	"net/http"
	"strconv"
	"time"

	"github.com/labstack/echo/v4"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/security/xray/internal/servers/humanizer/auth"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/controllers"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/infra"
	"a.yandex-team.ru/security/xray/internal/stagehealth"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	stagesLimit = 50
)

var _ controllers.Controller = (*Controller)(nil)

type Controller struct {
	*infra.Infra
}

func (c *Controller) BuildRoute(g *echo.Group) {
	authMiddleware := auth.NewAuthMiddleware(c.Infra)

	apiG := g.Group("/api/v1", corsMiddleware, authMiddleware)
	apiG.GET("/stage/list", c.listStages)
	apiG.GET("/stage/list/all", c.listAllStages)
	apiG.GET("/stage/:id/:uuid/:revision/status", c.stageStatus)
}

func (c *Controller) listStages(e echo.Context) error {
	rsp, err := c.XRay.List(
		auth.WithXrayCredentials(c.Infra, e),
		&xrayrpc.StageListRequest{Limit: stagesLimit})
	if err != nil {
		return err
	}

	return e.JSON(http.StatusOK, rsp.Statuses)
}

func (c *Controller) listAllStages(e echo.Context) error {
	rsp, err := c.XRay.ListAll(
		auth.WithXrayCredentials(c.Infra, e),
		&xrayrpc.StageListAllRequest{Limit: stagesLimit})
	if err != nil {
		return err
	}

	return e.JSON(http.StatusOK, rsp.Statuses)
}

func (c *Controller) stageStatus(e echo.Context) error {
	stageRevision, err := strconv.ParseUint(e.Param("revision"), 10, 32)
	if err != nil {
		return fmt.Errorf("failed to parse revision: %w", err)
	}

	if stageRevision == 0 {
		return fmt.Errorf("empty revision")
	}

	stageID := e.Param("id")
	if stageID == "" {
		return fmt.Errorf("empty stage id")
	}

	stageUUID := e.Param("uuid")
	if stageUUID == "" {
		return fmt.Errorf("empty stage uuid")
	}

	rsp, err := c.XRay.GetStatus(
		auth.WithXrayCredentials(c.Infra, e),
		&xrayrpc.StageGetStatusRequest{
			Stage: &xrayrpc.Stage{
				Id:       stageID,
				Uuid:     stageUUID,
				Revision: uint32(stageRevision),
			},
		})
	if err != nil {
		st, ok := status.FromError(err)
		if ok && st.Code() == codes.NotFound {
			// probably stage not scheduled yet
			out := StageStatusRsp{
				AnalysisStatusID: xrayrpc.AnalysisStatusKind_ASK_UNKNOWN,
				AnalysisStatus:   statusName(xrayrpc.AnalysisStatusKind_ASK_UNKNOWN),
				StageHealthID:    xrayrpc.StageHealthKind_SHK_UNKNOWN,
				StageHealth:      stagehealth.Name(xrayrpc.StageHealthKind_SHK_UNKNOWN),
				UpdatedAt:        time.Now(),
			}
			return e.JSON(http.StatusOK, out)
		}
		return err
	}

	out := StageStatusRsp{
		AnalysisStatusID: rsp.Status.AnalysisStatus,
		AnalysisStatus:   statusName(rsp.Status.AnalysisStatus),
		StageHealthID:    rsp.Status.StageHealth,
		StageHealth:      stagehealth.Name(rsp.Status.StageHealth),
		UpdatedAt:        rsp.Status.UpdatedAt.AsTime(),
		Issues: StageStatusIssues{
			Low:    rsp.Status.Issues.Low,
			Medium: rsp.Status.Issues.Medium,
			High:   rsp.Status.Issues.High,
		},
	}
	return e.JSON(http.StatusOK, out)
}
