package pages

import (
	"fmt"
	"net/http"
	"path"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/xray/internal/servers/humanizer/auth"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/controllers"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/infra"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	stagesLimit = 50
)

var _ controllers.Controller = (*Controller)(nil)

type Controller struct {
	*infra.Infra
}

func (c *Controller) BuildRoute(g *echo.Group) {
	authMiddleware := auth.NewAuthMiddleware(c.Infra)

	g.GET("/", c.redirect)
	g.GET("/my", c.listMy, authMiddleware)
	g.GET("/all", c.listAll, authMiddleware)
	g.GET("/stage/:id/:uuid/:revision", c.stage, authMiddleware)
}

func (c *Controller) redirect(e echo.Context) error {
	return e.Redirect(http.StatusTemporaryRedirect, path.Join(e.Path(), "my"))
}

func (c *Controller) listAll(e echo.Context) error {
	rsp, err := c.XRay.ListAll(
		auth.WithXrayCredentials(c.Infra, e),
		&xrayrpc.StageListAllRequest{Limit: stagesLimit})
	if err != nil {
		return err
	}

	return e.Render(http.StatusOK, "stages.html", echo.Map{
		"Title":    "All stages",
		"Statuses": rsp.Statuses,
	})
}

func (c *Controller) listMy(e echo.Context) error {
	rsp, err := c.XRay.List(
		auth.WithXrayCredentials(c.Infra, e),
		&xrayrpc.StageListRequest{Limit: stagesLimit})
	if err != nil {
		return err
	}

	return e.Render(http.StatusOK, "stages.html", echo.Map{
		"Title":    "My stages",
		"Statuses": rsp.Statuses,
	})
}

func (c *Controller) stage(e echo.Context) error {
	stageRevision, err := strconv.ParseUint(e.Param("revision"), 10, 32)
	if err != nil {
		return fmt.Errorf("failed to parse revision: %w", err)
	}

	if stageRevision == 0 {
		return fmt.Errorf("empty revision")
	}

	stageID := e.Param("id")
	if stageID == "" {
		return fmt.Errorf("empty stage id")
	}

	stageUUID := e.Param("uuid")
	if stageUUID == "" {
		return fmt.Errorf("empty stage uuid")
	}

	rsp, err := c.XRay.GetResults(
		auth.WithXrayCredentials(c.Infra, e),
		&xrayrpc.Stage{
			Id:       stageID,
			Uuid:     stageUUID,
			Revision: uint32(stageRevision),
		})
	if err != nil {
		return err
	}

	if rsp.AnalysisStatus != xrayrpc.AnalysisStatusKind_ASK_DONE {
		return e.Render(http.StatusOK, "stage-incomplete.html", echo.Map{
			"StageID":       stageID,
			"StageRevision": stageRevision,
			"Results":       StageIncompleteResultsFromProto(rsp),
		})
	}

	stageResults := StageResultsFromProto(rsp, e.QueryParam("severity"))
	return e.Render(http.StatusOK, "stage.html", echo.Map{
		"StageID":       stageID,
		"StageRevision": stageRevision,
		"Results":       stageResults,
	})
}
