package pages

import (
	"bytes"
	"errors"
	"fmt"
	"html/template"
	"sort"
	"strings"

	"github.com/yuin/goldmark"
	"github.com/yuin/goldmark/extension"
	"github.com/yuin/goldmark/renderer"
	"github.com/yuin/goldmark/util"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/security/libs/go/goldmark/yrenderer"
	"a.yandex-team.ru/security/xray/pkg/checks"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

type FiltersSet struct {
	Kind     map[string]string
	Target   map[string]string
	Severity map[string]string
	Type     map[string]string
}

type Filter struct {
	ID   string
	Name string
}

type Filters struct {
	Target   []Filter
	Severity []Filter
	Type     []Filter
	Kind     []Filter
}

type StageResults struct {
	AnalyzeID         string
	UpdatedAt         *timestamppb.Timestamp
	Issues            []Issue
	Warnings          []Warning
	Filters           Filters
	LogURI            string
	AnalysisStatus    xrayrpc.AnalysisStatusKind
	StageHealth       xrayrpc.StageHealthKind
	StatusDescription string
}

type IncompleteStageResults struct {
	AnalyzeID         string
	UpdatedAt         *timestamppb.Timestamp
	Warnings          []Warning
	LogURI            string
	AnalysisStatus    xrayrpc.AnalysisStatusKind
	StatusDescription string
}

type Issue struct {
	Target       string
	Type         string
	TypeName     string
	Kind         string
	ID           string
	Severity     string
	SeverityName string
	Summary      string
	HelpURL      string
	Reference    string
	Description  template.HTML
}

type Warning struct {
	Target  string
	Message string
}

var (
	md = goldmark.New(
		goldmark.WithExtensions(extension.GFM),
		goldmark.WithRendererOptions(
			renderer.WithNodeRenderers(
				util.Prioritized(yrenderer.NewFixedHeading(3), 0),
			),
		),
	)
)

func StageIncompleteResultsFromProto(proto *xrayrpc.StageGetResultsReply) IncompleteStageResults {
	return IncompleteStageResults{
		AnalyzeID:         strings.SplitN(proto.AnalyzeId, ":", 2)[0],
		UpdatedAt:         proto.UpdatedAt,
		LogURI:            proto.LogUri,
		AnalysisStatus:    proto.AnalysisStatus,
		StatusDescription: proto.StatusDescription,
		Warnings:          formatWarnings(proto.Results.Warnings),
	}
}

func StageResultsFromProto(proto *xrayrpc.StageGetResultsReply, rawSeverity string) StageResults {
	severity := parseSeverity(rawSeverity)
	filterSet := FiltersSet{
		Type:     make(map[string]string),
		Severity: make(map[string]string),
		Target:   make(map[string]string),
		Kind:     make(map[string]string),
	}

	issues := formatResults(proto.Results.Issues, &filterSet, severity)
	// fist of all - sort by target
	sort.SliceStable(issues, func(i, k int) bool {
		return issues[i].Target < issues[k].Target
	})

	// them - by type
	sort.SliceStable(issues, func(i, k int) bool {
		return issues[i].Type < issues[k].Type
	})

	// and finally - by severity
	sort.SliceStable(issues, func(i, k int) bool {
		return issues[i].Severity < issues[k].Severity
	})

	return StageResults{
		AnalyzeID:         strings.SplitN(proto.AnalyzeId, ":", 2)[0],
		UpdatedAt:         proto.UpdatedAt,
		LogURI:            proto.LogUri,
		AnalysisStatus:    proto.AnalysisStatus,
		StageHealth:       proto.StageHealth,
		StatusDescription: proto.StatusDescription,
		Issues:            issues,
		Filters:           filterSet.Filters(),
		Warnings:          formatWarnings(proto.Results.Warnings),
	}
}

func formatResults(issues []*xrayrpc.Issue, filters *FiltersSet, severity xrayrpc.Severity) []Issue {
	out := make([]Issue, 0, len(issues))
	for _, issue := range issues {
		if issue.Severity < severity {
			continue
		}

		issueType, formattedIssue, err := checks.FormatIssue(issue)
		if err != nil {
			if errors.Is(err, check.ErrUnknownKind) {
				continue
			}
			panic(fmt.Sprintf("failed to format issue: %s", err))
		}

		issueKind := issue.Kind.String()
		if _, ok := filters.Kind[issueKind]; !ok {
			filters.Kind[issueKind] = issueKindName(issue.Kind)
		}

		issueTypeName := checks.CheckName(issueType)
		target := formatTarget(issue.Target)
		if _, ok := filters.Target[target]; !ok {
			filters.Target[target] = target
		}

		severity, severityName := formatSeverity(issue.Severity)
		if _, ok := filters.Severity[severity]; !ok {
			filters.Severity[severity] = severityName
		}

		if _, ok := filters.Type[issueType]; !ok {
			filters.Type[issueType] = issueTypeName
		}

		desc, err := renderMD(formattedIssue.Description)
		if err != nil {
			desc = fmt.Sprintf("failed to render description: %s", err)
		}

		analyseResult := Issue{
			Target:       target,
			Type:         issueType,
			TypeName:     issueTypeName,
			Kind:         issueKind,
			ID:           formattedIssue.ID,
			Severity:     severity,
			SeverityName: severityName,
			Summary:      formattedIssue.Summary,
			HelpURL:      formattedIssue.HelpURL,
			Reference:    formattedIssue.Reference,
			Description:  template.HTML(desc),
		}

		out = append(out, analyseResult)
	}
	return out
}

func formatWarnings(warnings []*xrayrpc.Warning) []Warning {
	out := make([]Warning, 0, len(warnings))
	for _, warning := range warnings {
		if warning.Message == "" {
			continue
		}

		out = append(out, Warning{
			Target:  formatTarget(warning.Target),
			Message: warning.Message,
		})
	}
	return out
}

func formatTarget(target *xrayrpc.TargetPath) string {
	var path []string
	parent := target
	for ; parent != nil; parent = parent.Parent {
		path = append([]string{parent.Name}, path...)
	}

	return strings.Join(path, ".")
}

func parseSeverity(severity string) xrayrpc.Severity {
	switch severity {
	case "1":
		return xrayrpc.Severity_S_INFO
	case "2":
		return xrayrpc.Severity_S_LOW
	case "3":
		return xrayrpc.Severity_S_MEDIUM
	case "4":
		return xrayrpc.Severity_S_HIGH
	default:
		return xrayrpc.Severity_S_LOW
	}
}

func formatSeverity(severity xrayrpc.Severity) (string, string) {
	switch severity {
	case xrayrpc.Severity_S_INFO:
		return "4-info", "Info"
	case xrayrpc.Severity_S_LOW:
		return "3-low", "Low"
	case xrayrpc.Severity_S_MEDIUM:
		return "2-medium", "Medium"
	case xrayrpc.Severity_S_HIGH:
		return "1-high", "High"
	default:
		return "4-info", "Info"
	}
}

func issueKindName(kind xrayrpc.IssueKind) string {
	switch kind {
	case xrayrpc.IssueKind_IK_SECURITY:
		return "Security"
	case xrayrpc.IssueKind_IK_AVAILABILITY:
		return "Availability"
	default:
		panic(fmt.Sprintf("unknown issue kind: %s", kind))
	}
}

func (f *FiltersSet) Filters() Filters {
	toSlice := func(in map[string]string) []Filter {
		out := make([]Filter, 0, len(in))
		for id, name := range in {
			out = append(out, Filter{ID: id, Name: name})
		}

		sort.Slice(out, func(i, j int) bool {
			return out[i].ID < out[j].ID
		})
		return out
	}

	return Filters{
		Target:   toSlice(f.Target),
		Severity: toSlice(f.Severity),
		Type:     toSlice(f.Type),
		Kind:     toSlice(f.Kind),
	}
}

func renderMD(source string) (string, error) {
	var buf bytes.Buffer
	if err := md.Convert([]byte(source), &buf); err != nil {
		return "", err
	}
	return buf.String(), nil
}
