package echoutils

import (
	"net/http"
	"os"
	"path"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/httputil/resource"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/infra"
)

func NewHTTPErrorHandler(i *infra.Infra) func(err error, c echo.Context) {
	return func(err error, c echo.Context) {
		i.Logger.Error("request error", log.String("uri", c.Request().URL.String()), log.Error(err))

		var (
			code = http.StatusInternalServerError
			msg  interface{}
		)

		if he, ok := err.(*echo.HTTPError); ok {
			code = he.Code
			msg = he.Message
		} else {
			msg = err.Error()
		}

		if _, ok := msg.(string); ok {
			msg = echo.Map{"error": msg}
		} else {
			msg = echo.Map{"error": "unknown"}
		}

		// Send response
		if !c.Response().Committed {
			if c.Request().Method == http.MethodHead { // Issue #608
				_ = c.NoContent(code)
			} else {
				_ = c.JSON(code, msg)
			}
		}
	}
}

func NewResourceStatic(prefix, resourceDir string) func(c echo.Context) error {
	dir := resource.Dir(resourceDir)
	return func(c echo.Context) error {
		filename := strings.TrimPrefix(path.Clean(c.Request().URL.Path), prefix)
		f, err := dir.Open(filename)
		switch err {
		case nil:
			// pass
		case os.ErrNotExist:
			return c.String(http.StatusNotFound, "file not found")
		default:
			return err
		}

		stat, _ := f.Stat()
		c.Response().Header().Set("Cache-Control", "public,max-age=86400")
		http.ServeContent(c.Response(), c.Request(), filename, stat.ModTime(), f)
		_ = f.Close()
		return nil
	}
}
