package infra

import (
	"context"
	"fmt"
	"os"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/security/libs/go/xtvm"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/config"
	"a.yandex-team.ru/security/xray/pkg/xray"
)

const tvmWaitTimeout = 30 * time.Second

type Infra struct {
	Config          *config.Config
	TVM             tvm.Client
	BlackBox        blackbox.Client
	XRay            *xray.Client
	XRayCredentials xray.TVMCredentials
	Logger          log.Logger
}

func New(cfg *config.Config) *Infra {
	zlog, err := zap.NewDeployLogger(cfg.LogLvl.Level)
	if err != nil {
		panic(fmt.Sprintf("failed to create logger: %s", err))
	}

	return &Infra{
		Config: cfg,
		Logger: zlog,
	}
}

func (c *Infra) Initialize() (err error) {
	c.TVM, err = c.newTVMClient()
	if err != nil {
		return fmt.Errorf("create TVM client: %w", err)
	}

	if err := xtvm.Wait(c.TVM, tvmWaitTimeout); err != nil {
		return fmt.Errorf("wait tvm: %w", err)
	}

	c.BlackBox, err = httpbb.NewIntranet(
		httpbb.WithTVM(c.TVM),
		httpbb.WithLogger(c.Logger.Structured()),
	)
	if err != nil {
		return fmt.Errorf("create Blackbox client: %w", err)
	}

	c.XRayCredentials = xray.TVMCredentials{
		TVMClient: c.TVM,
		ClientID:  c.Config.Xray.TvmID,
	}

	c.XRay, err = xray.NewClient(
		xray.WithAddr(c.Config.Xray.Addr),
		xray.WithInsecure(c.Config.Xray.Insecure),
		xray.WithLogger(c.Logger.Structured()),
		xray.WithTimeout(1*time.Second),
	)
	if err != nil {
		return fmt.Errorf("create X-Ray client: %w", err)
	}

	return
}

func (c *Infra) Finish(_ context.Context) error {
	return nil
}

func (c *Infra) newTVMClient() (tvm.Client, error) {
	if c.Config.TVM.CacheDir != "" {
		if err := os.MkdirAll(c.Config.TVM.CacheDir, 0o700); err != nil {
			return nil, fmt.Errorf("unable to create tvm cache dir: %w", err)
		}
	}

	tvmSettings := tvmauth.TvmAPISettings{
		SelfID:                      c.Config.TVM.ClientID,
		ServiceTicketOptions:        tvmauth.NewAliasesOptions(c.Config.TVM.ClientSecret, c.Config.TVM.Destinations),
		DiskCacheDir:                c.Config.TVM.CacheDir,
		BlackboxEnv:                 &c.Config.TVM.Env,
		DisableDefaultUIDCheck:      true,
		DisableSrcCheck:             true,
		EnableServiceTicketChecking: true,
	}

	if c.Config.TVM.Port != 0 {
		tvmSettings.TVMPort = c.Config.TVM.Port
	}

	return tvmauth.NewAPIClient(tvmSettings, c.Logger)
}
