package humanizer

import (
	"context"
	"fmt"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/controllers"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/controllers/api"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/controllers/common"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/controllers/pages"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/echoutils"
	"a.yandex-team.ru/security/xray/internal/servers/humanizer/infra"
)

type (
	Server struct {
		echo        *echo.Echo
		i           *infra.Infra
		controllers []controllers.Controller
	}
)

func New(infra *infra.Infra) *Server {
	return &Server{
		echo: echo.New(),
		i:    infra,
	}
}

func (s *Server) onStart() (err error) {
	if err := s.i.Initialize(); err != nil {
		return fmt.Errorf("failed to initialize infra: %w", err)
	}

	s.echo.HTTPErrorHandler = echoutils.NewHTTPErrorHandler(s.i)
	if s.echo.Renderer, err = newTemplateRenderer(); err != nil {
		return fmt.Errorf("failed to initialize renderer: %w", err)
	}

	s.controllers = []controllers.Controller{
		&pages.Controller{Infra: s.i},
		&common.Controller{Infra: s.i},
		&api.Controller{Infra: s.i},
	}
	return nil
}

func (s *Server) onEnd() (err error) {
	return s.i.Finish(context.Background())
}

func (s *Server) ListenAndServe() error {
	err := s.onStart()
	if err != nil {
		return fmt.Errorf("failed to start server: %w", err)
	}

	defer func() {
		err := s.onEnd()
		if err != nil {
			simplelog.Error("failed to stop", "err", err)
		}
	}()

	s.echo.Use(middleware.RecoverWithConfig(middleware.RecoverConfig{
		Skipper:           middleware.DefaultSkipper,
		StackSize:         4 << 10, // 4 KB
		DisableStackAll:   true,
		DisablePrintStack: false,
	}))

	// pages
	root := s.echo.Group("")
	for _, c := range s.controllers {
		c.BuildRoute(root)
	}

	root.GET("/static/*", echoutils.NewResourceStatic(
		"/static/",
		"resfs/file/security/xray/assets/humanizer/public",
	))

	s.i.Logger.Info("starting humanizer", log.String("addr", s.i.Config.Addr))
	return s.echo.Start(s.i.Config.Addr)
}

func (s *Server) Shutdown(ctx context.Context) error {
	return s.echo.Shutdown(ctx)
}
