package humanizer

import (
	"fmt"
	"html/template"
	"io"
	"path"
	"strings"
	"sync"
	"time"

	"github.com/labstack/echo/v4"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/resource"
	"a.yandex-team.ru/security/xray/internal/stagehealth"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	layoutName      = "layout.html"
	templatesPrefix = "resfs/file/security/xray/assets/humanizer/templates/"
	timeFormat      = "02 Jan 06 15:04"
)

type (
	TemplateRenderer struct {
		templates  sync.Map
		layoutTmpl []byte
	}
)

var (
	mskZone = func() *time.Location {
		l, err := time.LoadLocation("Europe/Moscow")
		if err != nil {
			panic(fmt.Sprintf("failed to locad MSK location: %s", err))
		}
		return l
	}()

	tmplFuncs = template.FuncMap{
		"capitalize": func(str string) string {
			return strings.ToUpper(str[:1]) + str[1:]
		},
		"formatAnalysisStatus": func(status xrayrpc.AnalysisStatusKind) string {
			switch status {
			case xrayrpc.AnalysisStatusKind_ASK_START:
				return "in progress"
			case xrayrpc.AnalysisStatusKind_ASK_DONE:
				return "done"
			case xrayrpc.AnalysisStatusKind_ASK_FAIL:
				return "failed"
			case xrayrpc.AnalysisStatusKind_ASK_ABORT:
				return "aborted"
			default:
				return "unknown"
			}
		},
		"formatProtoTimestamp": func(timestamp *timestamppb.Timestamp) string {
			return timestamp.AsTime().In(mskZone).Format(timeFormat)
		},
		"formatStageHealth": func(val xrayrpc.StageHealthKind) string {
			return stagehealth.Name(val)
		},
	}
)

func newTemplateRenderer() (*TemplateRenderer, error) {
	return &TemplateRenderer{
		layoutTmpl: resource.MustGet(path.Join(templatesPrefix, layoutName)),
	}, nil
}

func (t *TemplateRenderer) Render(w io.Writer, name string, data interface{}, _ echo.Context) error {
	tmpl, err := t.templateByName(name)
	if err != nil {
		return err
	}

	return tmpl.ExecuteTemplate(w, layoutName, data)
}

func (t *TemplateRenderer) templateByName(name string) (*template.Template, error) {
	cachedTmpl, ok := t.templates.Load(name)
	if ok {
		return cachedTmpl.(*template.Template), nil
	}

	var (
		tmpl = template.New(name).Funcs(tmplFuncs)
		err  error
	)

	tmpl, err = tmpl.Parse(string(resource.MustGet(path.Join(templatesPrefix, name))))
	if err != nil {
		return nil, fmt.Errorf("failed to parse template %s: %w", name, err)
	}

	tmpl, err = tmpl.Parse(string(resource.MustGet(path.Join(templatesPrefix, layoutName))))
	if err != nil {
		return nil, fmt.Errorf("failed to parse layout template: %w", err)
	}

	t.templates.Store(name, tmpl)
	return tmpl, nil
}
