package watcher

import (
	"context"
	"errors"
	"fmt"
	"time"

	"github.com/cenkalti/backoff/v4"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/security/libs/go/xtvm"
	"a.yandex-team.ru/security/xray/internal/servers/watcher/config"
	"a.yandex-team.ru/security/xray/pkg/xray"
	"a.yandex-team.ru/security/xray/pkg/xrayerrors"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/yp/go/yp"
	"a.yandex-team.ru/yp/go/yp/yperrors"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/yt/go/yterrors"
	"a.yandex-team.ru/yt/go/ytlock"
)

const (
	tvmWaitTimeout = 30 * time.Second
	backoffTimeout = 10 * time.Second
	watchPeriod    = 1 * time.Second
	ypEventsCount  = 50
)

type StageInfo struct {
	Revision  uint32
	AccountID string
	UUID      string
}

type Watcher struct {
	ctx          context.Context
	exclAccounts []string
	exitChan     chan struct{}
	shutDownFunc context.CancelFunc
	cache        *StageCache
	cfg          *config.Config
	tvm          *tvmtool.Client
	xray         *xray.Client
	yp           *yp.Client
	yt           yt.Client
	log          log.Logger
}

func NewWatcher(cfg *config.Config, l log.Logger) (*Watcher, error) {
	ctx, shutdown := context.WithCancel(context.Background())
	return &Watcher{
		ctx:          ctx,
		shutDownFunc: shutdown,
		exitChan:     make(chan struct{}),
		cache:        NewStageCache(),
		cfg:          cfg,
		log:          l,
		exclAccounts: cfg.ExclAccounts,
	}, nil
}

func (w *Watcher) onStart() (err error) {
	w.yp, err = yp.NewClient(
		"xdc",
		yp.WithAuthToken(w.cfg.YP.Token),
		yp.WithLogger(w.log),
	)
	if err != nil {
		return fmt.Errorf("create YP client: %w", err)
	}

	w.yt, err = ythttp.NewClient(&yt.Config{
		Proxy: w.cfg.YT.Proxy,
		Token: w.cfg.YT.Token,
		// disable yt logging due to many errs messages due to concurrent locking
		//Logger: w.log.Structured(),
	})
	if err != nil {
		return fmt.Errorf("create YT client: %w", err)
	}

	w.tvm, err = tvmtool.NewAnyClient(
		tvmtool.WithSrc(w.cfg.TvmSource),
		tvmtool.WithLogger(w.log.Structured()),
	)
	if err != nil {
		return fmt.Errorf("create TVM client: %w", err)
	}

	if err := xtvm.Wait(w.tvm, tvmWaitTimeout); err != nil {
		return fmt.Errorf("wait tvm: %w", err)
	}

	w.xray, err = xray.NewClient(
		xray.WithLogger(w.log.Structured()),
		xray.WithTvmAuth(w.tvm, w.cfg.Xray.TvmID),
		xray.WithAddr(w.cfg.Xray.Addr),
		xray.WithInsecure(w.cfg.Xray.Insecure),
	)
	if err != nil {
		return fmt.Errorf("X-Ray client: %w", err)
	}

	return nil
}

func (w *Watcher) onEnd() error {
	w.cache.Close()
	return nil
}

func (w *Watcher) Start() error {
	defer close(w.exitChan)

	err := w.onStart()
	if err != nil {
		return fmt.Errorf("failed to start watcher: %w", err)
	}

	w.log.Info("watcher started")

	defer func() {
		err := w.onEnd()
		if err != nil {
			w.log.Error("failed to stop watcher", log.Error(err))
		}
	}()

	lock := ytlock.NewLockOptions(w.yt, ypath.Path(w.cfg.YT.Path).JoinChild("lock"), ytlock.Options{
		CreateIfMissing: true,
		LockMode:        yt.LockExclusive,
	})

	err = backoff.RetryNotify(
		func() error {
			tx, err := lock.Acquire(w.ctx)
			if err != nil {
				if w.ctx.Err() != nil {
					return backoff.Permanent(err)
				}

				return err
			}
			defer func() { _ = lock.Release(w.ctx) }()

			err = w.watch(tx)
			if err != nil {
				if w.ctx.Err() != nil {
					return backoff.Permanent(err)
				}
				return err
			}

			return nil
		},
		backoff.WithContext(backoff.NewConstantBackOff(backoffTimeout), w.ctx),
		func(err error, sleep time.Duration) {
			if w.ctx.Err() != nil {
				// that's fine
				return
			}

			if yterrors.ContainsErrorCode(err, yterrors.CodeConcurrentTransactionLockConflict) {
				w.log.Info("failed to acquire YT lock", log.String("error", err.Error()))
				return
			}

			w.log.Error("watch failed", log.Duration("sleep", sleep), log.Error(err))
		},
	)

	if err != nil && w.ctx.Err() == nil {
		return fmt.Errorf("failed to watch stages: %w", err)
	}
	w.log.Info("watcher stopped")
	return nil
}

func (w *Watcher) Shutdown(ctx context.Context) error {
	w.shutDownFunc()

	// grateful wait processor wg
	select {
	case <-w.exitChan:
		// completed normally
		return nil
	case <-ctx.Done():
		// timed out
		return fmt.Errorf("time out")
	}
}

func (w *Watcher) watch(lostTx <-chan struct{}) error {
	meta, err := w.restoreMeta()
	if err != nil {
		return fmt.Errorf("failed to restore meta: %w", err)
	}

	for {
		select {
		case <-w.ctx.Done():
			// that's fine
			w.log.Info("watcher context canceled")
			return nil
		case <-lostTx:
			return errors.New("lost YT lock")
		case <-time.After(watchPeriod):
			// ok
		}

		nextTS, err := w.yp.GenerateTimestamp(w.ctx)
		if err != nil {
			return fmt.Errorf("failed to generate YP timestamp: %w", err)
		}

		var ypContinuationToken string
		for {
			req := yp.WatchStagesRequest{
				Timestamp:       nextTS,
				EventCountLimit: ypEventsCount,
				TimeLimit:       10 * time.Second,
			}

			if ypContinuationToken != "" {
				req.ContinuationToken = ypContinuationToken
			} else {
				req.StartTimestamp = meta.StartTimestamp
			}

			wr, err := w.yp.WatchStages(w.ctx, req)
			if err != nil {
				regen := yterrors.ContainsErrorCode(err, yperrors.RowsAlreadyTrimmed) || yterrors.ContainsErrorCode(err, yperrors.TimestampOutOfRange)
				if regen {
					ypContinuationToken = ""
					w.log.Warn("yp start timestamp are too old, regenerate it", log.UInt64("start_timestamp", meta.StartTimestamp))
					meta, err = w.createMeta()
					if err != nil {
						return fmt.Errorf("failed to create meta: %w", err)
					}
					continue
				}
				return fmt.Errorf("failed to watch stages: %w", err)
			}

			if len(wr.Events) > 0 {
				w.log.Info("new stage events",
					log.Int("events", len(wr.Events)),
				)
			}

			w.processEvents(wr.Events)

			if len(wr.Events) < ypEventsCount {
				// end reached
				break
			}
			ypContinuationToken = wr.ContinuationToken
		}

		meta.StartTimestamp = nextTS
		if err := w.saveMeta(meta); err != nil {
			w.log.Error("failed to save meta", log.Error(err))
		}
	}
}

func (w *Watcher) processEvents(events []yp.WatchObjectsEvent) {
	for _, event := range events {
		switch event.EventType {
		case yp.EventTypeObjectRemoved:
			w.cache.DeleteStage(event.ObjectID)
			fallthrough
		case yp.EventTypeObjectNone:
			continue
		case yp.EventTypeObjectCreated, yp.EventTypeObjectUpdated:
			// pass
		default:
			w.log.Error("unexpected event type", log.String("event_type", event.EventType.String()))
			continue
		}

		sr, err := w.yp.GetStage(w.ctx, yp.GetStageRequest{
			Timestamp: event.Timestamp,
			Format:    yp.PayloadFormatYson,
			ID:        event.ObjectID,
			Selectors: []string{"/spec/revision", "/spec/account_id", "/meta/uuid"},
		})
		if err != nil {
			w.log.Error("failed to get stage info",
				log.UInt64("timestamp", event.Timestamp),
				log.String("stage_id", event.ObjectID),
				log.Error(err),
			)
			continue
		}

		var stageInfo StageInfo
		err = sr.Fill(
			&stageInfo.Revision,
			&stageInfo.AccountID,
			&stageInfo.UUID,
		)
		if err != nil {
			w.log.Error("failed to fill stage info",
				log.UInt64("timestamp", event.Timestamp),
				log.String("stage_id", event.ObjectID),
				log.Error(err),
			)
			continue
		}

		skip := false
		for _, excl := range w.exclAccounts {
			if stageInfo.AccountID == excl {
				w.log.Info("skip excluded project",
					log.UInt64("timestamp", event.Timestamp),
					log.String("stage_id", event.ObjectID),
					log.String("account_id", stageInfo.AccountID),
				)
				skip = true
				break
			}
		}

		if skip {
			continue
		}

		stageID := event.ObjectID
		cachedLastRev := w.cache.GetStageRevision(stageID)
		if cachedLastRev > 0 && cachedLastRev >= stageInfo.Revision {
			w.log.Info("skip old revision",
				log.UInt64("timestamp", event.Timestamp),
				log.String("stage_id", stageID),
				log.UInt32("stage_revision", stageInfo.Revision),
				log.UInt32("latest_revision", cachedLastRev),
			)
			continue
		}

		_, err = w.xray.Schedule(w.ctx, &xrayrpc.StageScheduleRequest{
			Force: false,
			Stage: &xrayrpc.Stage{
				Id:       event.ObjectID,
				Uuid:     stageInfo.UUID,
				Revision: stageInfo.Revision,
			},
		})

		if err != nil {
			if grpcErr, ok := status.FromError(err); ok && grpcErr.Code() == xrayerrors.ErrCodeConflictSchedule {
				// that's fine
				w.log.Info("stage analyze conflict",
					log.UInt64("timestamp", event.Timestamp),
					log.String("stage_id", event.ObjectID),
					log.String("stage_uuid", stageInfo.UUID),
					log.UInt32("stage_revision", stageInfo.Revision),
				)
				w.cache.SetStageRevision(stageID, stageInfo.Revision)
			} else {
				w.log.Error("failed to schedule stage info",
					log.UInt64("timestamp", event.Timestamp),
					log.String("stage_id", event.ObjectID),
					log.String("stage_uuid", stageInfo.UUID),
					log.UInt32("stage_revision", stageInfo.Revision),
					log.Error(err),
				)
			}
			continue
		}

		w.cache.SetStageRevision(stageID, stageInfo.Revision)
		w.log.Info("stage analyze scheduled",
			log.String("stage_id", event.ObjectID),
			log.String("stage_uuid", stageInfo.UUID),
			log.UInt32("stage_revision", stageInfo.Revision),
		)
	}
}

func (w *Watcher) createMeta() (*Meta, error) {
	ts, err := w.yp.GenerateTimestamp(w.ctx)
	if err != nil {
		return nil, fmt.Errorf("failed to generate YP timestamp: %w", err)
	}

	return &Meta{
		StartTimestamp: ts,
	}, nil
}

func (w *Watcher) restoreMeta() (*Meta, error) {
	metaPath := ypath.Path(w.cfg.YT.Path).JoinChild("meta")
	var meta Meta
	err := w.yt.GetNode(w.ctx, metaPath, &meta, nil)
	if err == nil {
		return &meta, nil
	}

	w.log.Error("failed to restore meta from YT, create new one", log.Error(err))
	ts, err := w.yp.GenerateTimestamp(w.ctx)
	if err != nil {
		return nil, fmt.Errorf("failed to generate YP timestamp: %w", err)
	}

	return &Meta{
		StartTimestamp: ts,
	}, nil
}

func (w *Watcher) saveMeta(meta *Meta) error {
	metaPath := ypath.Path(w.cfg.YT.Path).JoinChild("meta")
	return w.yt.SetNode(context.Background(), metaPath, meta, &yt.SetNodeOptions{Force: true})
}
