package worker

import (
	"bytes"
	"context"
	"fmt"
	"os"
	"time"

	"github.com/gofrs/uuid"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	aZap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/security/xray/internal/dbmodels"
	"a.yandex-team.ru/security/xray/internal/servers/worker/results"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

var (
	uuidGen = uuid.NewGen()
)

func (w *Worker) saveResponse(ctx context.Context, result *results.Analyze) error {
	issuesInfo := dbmodels.StageIssuesCounter{}
	var warnings uint32
	if result.Results != nil {
		warnings = uint32(len(result.Results.Warnings))
		for _, issue := range result.Results.Issues {
			switch issue.Severity {
			case xrayrpc.Severity_S_UNKNOWN:
				issuesInfo.Unknown++
			case xrayrpc.Severity_S_INFO:
				issuesInfo.Info++
			case xrayrpc.Severity_S_LOW:
				issuesInfo.Low++
			case xrayrpc.Severity_S_MEDIUM:
				issuesInfo.Medium++
			case xrayrpc.Severity_S_HIGH:
				issuesInfo.High++
			default:
				w.log.Error("unknown severity", log.String("analyze_id", result.AnalyzeID))
			}
		}
	}

	updateData := dbmodels.UpdateAnalyzeData{
		AnalyzeID:         result.AnalyzeID,
		StageUUID:         result.Stage.Uuid,
		StageRevision:     result.Stage.Revision,
		UpdatedAt:         time.Now(),
		ResultPath:        result.ResultPath,
		LogPath:           result.LogPath,
		Status:            result.Status,
		StatusDescription: result.StatusDescription,
		Overview: dbmodels.StageOverview{
			Issues:   issuesInfo,
			Warnings: warnings,
		},
	}

	err := w.db.CompleteAnalyze(ctx, updateData)
	if err != nil {
		return fmt.Errorf("save fail: %w", err)
	}
	return nil
}

func (w *Worker) uploadLog(ctx context.Context, analyzeID, logPath string) (string, error) {
	f, err := os.Open(logPath)
	if err != nil {
		return "", err
	}

	defer func() { _ = f.Close }()

	return w.s3Storage.UploadLog(ctx, analyzeID, f)
}

func (w *Worker) uploadResult(ctx context.Context, analyzeID string, result *results.Analyze) (string, error) {
	if result == nil || result.Results == nil || len(result.Results.Issues) == 0 && len(result.Results.Warnings) == 0 {
		// ok
		return "", nil
	}

	data, err := proto.Marshal(result.Results)
	if err != nil {
		return "", fmt.Errorf("failed to upload results: encode failed: %w", err)
	}

	return w.s3Storage.UploadResult(ctx, analyzeID, bytes.NewReader(data))
}

func (w *Worker) newProcessorLogger(outputPath string) (*aZap.Logger, error) {
	cfg := zap.Config{
		Level:            zap.NewAtomicLevelAt(aZap.ZapifyLevel(w.cfg.LogLvl.Level)),
		Encoding:         "console",
		OutputPaths:      []string{outputPath},
		ErrorOutputPaths: []string{"stderr"},
		DisableCaller:    true,
		Development:      false,
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			TimeKey:        "ts",
			EncodeLevel:    zapcore.CapitalLevelEncoder,
			EncodeTime:     zapcore.ISO8601TimeEncoder,
			EncodeDuration: zapcore.StringDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	}

	return aZap.New(cfg)
}
