package config

import (
	"context"
	"fmt"
	"os"
	"strings"
	"time"

	"github.com/c2h5oh/datasize"
	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/xray/internal/db"
	"a.yandex-team.ru/security/xray/internal/logutil"
	"a.yandex-team.ru/security/xray/internal/storage/s3storage"
)

type SQS struct {
	Endpoint string `yaml:"endpoint"`
	Account  string `yaml:"account"`
	Queue    string `yaml:"queue"`
}

type S3 struct {
	Endpoint    string       `yaml:"endpoint"`
	AccessKeyID string       `yaml:"access_key_id"`
	Bucket      string       `yaml:"bucket"`
	TvmID       tvm.ClientID `yaml:"tvm_id"`
}

type YDB struct {
	Endpoint string `yaml:"endpoint"`
	Database string `yaml:"database"`
	Path     string `yaml:"path"`
}

type Splunk struct {
	Enabled    bool   `yaml:"enabled"`
	Index      string `yaml:"index"`
	SourceType string `yaml:"source_type"`
	AuthToken  string `yaml:"auth_token"`
}

type FsStorage struct {
	Dir  string            `yaml:"dir"`
	Size datasize.ByteSize `yaml:"size"`
}

type Checks struct {
	HostDir      string `yaml:"host_dir"`
	ContainerDir string `yaml:"container_dir"`
	AuthToken    string `yaml:"auth_token"`
}

type Collectors struct {
	ContainerDir string `yaml:"container_dir"`
	HostDir      string `yaml:"host_dir"`
}

type TVM struct {
	ClientID     tvm.ClientID            `yaml:"tvm_id"`
	ClientSecret string                  `yaml:"-"`
	CacheDir     string                  `yaml:"cache_dir"`
	Destinations map[string]tvm.ClientID `yaml:"dests"`
	Env          tvm.BlackboxEnv         `yaml:"env"`
	Port         int                     `yaml:"port"`
}

type Worker struct {
	Concurrency     int               `yaml:"concurrency"`
	WorkDir         string            `yaml:"work_dir"`
	LayerStorage    FsStorage         `yaml:"layer_storage"`
	ResourceStorage FsStorage         `yaml:"resource_storage"`
	PortoOpts       map[string]string `yaml:"porto_opts"`
	Checks          Checks            `yaml:"checks"`
	Collectors      Collectors        `yaml:"collectors"`
	SyncPeriod      time.Duration     `yaml:"sync_period"`
}

type Config struct {
	LogLvl  logutil.Level `yaml:"log_level"`
	SQS     SQS           `yaml:"sqs"`
	S3      S3            `yaml:"s3"`
	YDB     YDB           `yaml:"ydb"`
	Splunk  Splunk        `yaml:"splunk"`
	Worker  Worker        `yaml:"worker"`
	YpToken string        `yaml:"yp_token"`
	SignKey string        `yaml:"sign_key"`
	TVM     TVM           `yaml:"tvm"`
}

func Load(cfgPath string) (*Config, error) {
	cfg := &Config{
		LogLvl: logutil.Level{
			Level: log.DebugLevel,
		},
		Worker: Worker{
			Checks: Checks{
				AuthToken: os.Getenv("CHECKS_TOKEN"),
			},
		},
		Splunk: Splunk{
			AuthToken: os.Getenv("SPLUNK_TOKEN"),
		},
		YpToken: os.Getenv("YP_TOKEN"),
		SignKey: os.Getenv("SIGN_KEY"),
		TVM: TVM{
			ClientSecret: os.Getenv("TVM_SECRET_ANY"),
		},
	}

	loader := confita.NewLoader(env.NewBackend(), file.NewBackend(cfgPath))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, err
	}

	return cfg, nil
}

func (c *Config) DBConfig() *db.Options {
	return &db.Options{
		Endpoint: c.YDB.Endpoint,
		Database: c.YDB.Database,
		Path:     c.YDB.Path,
	}
}

func (c *Config) S3StorageConfig() *s3storage.Options {
	return &s3storage.Options{
		Endpoint:    c.S3.Endpoint,
		Bucket:      c.S3.Bucket,
		AccessKeyID: c.S3.AccessKeyID,
		TvmID:       c.S3.TvmID,
	}
}

func (c *Config) RequestsQueueURL() string {
	return fmt.Sprintf("%s/%s/%s",
		strings.TrimRight(c.SQS.Account, "/"),
		c.SQS.Account,
		c.SQS.Queue,
	)
}
