package inspect

import (
	"bytes"
	"context"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"sort"
	"strconv"
	"strings"
	"sync"
	"syscall"
	"time"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/libs/go/porto"
	"a.yandex-team.ru/security/xray/internal/image-resolvers/dockinfo"
	"a.yandex-team.ru/security/xray/internal/storage/fetcher"
	"a.yandex-team.ru/security/xray/internal/storage/layerstorage"
	"a.yandex-team.ru/security/xray/internal/storage/rescache"
	"a.yandex-team.ru/security/xray/internal/storage/resstorage"
	"a.yandex-team.ru/security/xray/internal/storage/s3storage"
	"a.yandex-team.ru/security/xray/internal/stringutil"
	"a.yandex-team.ru/security/xray/internal/yputil"
	"a.yandex-team.ru/security/xray/pkg/checks"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/collectors"
	"a.yandex-team.ru/security/xray/pkg/collectors/collector"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/yp/go/yson/podagent"
	"a.yandex-team.ru/yp/go/yson/ypapi"
)

type (
	Inspector struct {
		l               log.Logger
		outL            log.Logger
		analyzeID       string
		workDir         string
		checks          checks.Checks
		collectors      collectors.Collectors
		portoOpts       map[string]string
		layerStorage    *layerstorage.Storage
		resourceStorage *resstorage.Storage
		s3Storage       *s3storage.Storage
		porto           *porto.API

		// runtime caches
		cachedLayers    []*rescache.Resource
		cachedResources []*rescache.Resource
		portoLayers     map[string]struct{}
		volumes         []*porto.Volume
	}

	container struct {
		collector.ContainerRequirements
		root     string
		cmd      string
		deadline time.Duration
	}
)

var baseRootFSFolders = []string{
	// w/o this folders porto doesn't start container with r/o fs on overlay fs volume %)
	"dev",
	"proc",
	"run",
	"sys",

	// w/o this folders our bind mount will fail
	"checks",
}

const containerStopTimeout = 5 * time.Second

func NewInspector(opts ...Option) *Inspector {
	inspector := &Inspector{
		l:           &nop.Logger{},
		outL:        &nop.Logger{},
		analyzeID:   "",
		workDir:     os.TempDir(),
		portoLayers: make(map[string]struct{}),
	}

	for _, opt := range opts {
		opt(inspector)
	}

	if inspector.checks == nil {
		panic("no checks configured for inspector")
	}

	return inspector
}

func (i *Inspector) Close() {
	i.cleanupVolumes()
	i.cleanupLayers()
	i.cleanupResources()
}

func (i *Inspector) Inspect(ctx context.Context, stage *ypapi.TStage) (*xrayrpc.AnalyzeResult, error) {
	result := new(xrayrpc.AnalyzeResult)
	err := i.inspectStage(ctx, stage, result)
	if err != nil {
		return result, err
	}

	// sort issues by severity in reverse order
	sort.Slice(result.Issues, func(i, j int) bool {
		return result.Issues[i].Severity > result.Issues[j].Severity
	})

	return result, nil
}

func (i *Inspector) inspectStage(ctx context.Context, stage *ypapi.TStage, result *xrayrpc.AnalyzeResult) error {
	stageID := stage.GetMeta().GetId()
	appL := log.With(i.l, log.String("stage_id", stageID))
	outL := log.With(i.outL, log.String("stage_id", stageID))

	appL.Info("inspect stage")
	outL.Info("inspect stage")

	sSpec := &check.StageSpec{
		Stage: stage,
		Path: &xrayrpc.TargetPath{
			Name: stage.GetMeta().GetId(),
			Kind: xrayrpc.TargetKind_TK_STAGE,
		},
	}

	for k := range i.checks[check.TargetKindStage] {
		ch := i.checks[check.TargetKindStage][k].(check.StageCheck)
		issues, err := ch.CheckStage(outL, sSpec)
		if err != nil {
			appL.Error("stage check fail",
				log.String("check_name", ch.Name()),
				log.Error(err))
			outL.Error("stage check fail",
				log.String("check_name", ch.Name()),
				log.Error(err))

			result.Warnings = append(result.Warnings, &xrayrpc.Warning{
				Target:  sSpec.Path,
				Message: err.Error(),
			})
			continue
		}

		outL.Info("stage check finished", log.String("check_name", ch.Name()))
		for _, issue := range issues {
			issue.Target = sSpec.Path
			result.Issues = append(result.Issues, issue)
		}
	}

	for deployUnitName, spec := range sSpec.Stage.Spec.DeployUnits {
		duSpec := &check.DeployUnitSpec{
			Stage: sSpec,
			Name:  deployUnitName,
			Spec:  spec,
			Path: &xrayrpc.TargetPath{
				Name:   deployUnitName,
				Kind:   xrayrpc.TargetKind_TK_DEPLOY_UNIT,
				Parent: sSpec.Path,
			},
		}

		if err := i.inspectDeployUnit(ctx, duSpec, result); err != nil {
			return fmt.Errorf("deploy unit %q analysis fail: %w", targetPathID(duSpec.Path), err)
		}
	}

	return nil
}

func (i *Inspector) inspectDeployUnit(ctx context.Context, deployUnit *check.DeployUnitSpec, result *xrayrpc.AnalyzeResult) error {
	duID := targetPathID(deployUnit.Path)
	appL := log.With(i.l, log.String("deploy_unit", duID))
	outL := log.With(i.outL, log.String("deploy_unit", duID))

	appL.Info("inspect deploy unit")
	outL.Info("inspect deploy unit")

	for k := range i.checks[check.TargetKindDeployUnit] {
		ch := i.checks[check.TargetKindDeployUnit][k].(check.DeployUnitCheck)
		issues, err := ch.CheckDeployUnit(outL, deployUnit)
		if err != nil {
			appL.Error("deploy unit check fail",
				log.String("check_name", ch.Name()),
				log.Error(err))
			outL.Error("deploy unit check fail",
				log.String("check_name", ch.Name()),
				log.Error(err))

			result.Warnings = append(result.Warnings, &xrayrpc.Warning{
				Target:  deployUnit.Path,
				Message: err.Error(),
			})
			continue
		}

		outL.Info("deploy unit check finished", log.String("check_name", ch.Name()))
		for _, issue := range issues {
			issue.Target = deployUnit.Path
			result.Issues = append(result.Issues, issue)
		}
	}

	podSpec := yputil.DeployUnitPodSpec(deployUnit.Spec)
	if podSpec == nil {
		outL.Info("skip boxes check",
			log.String("reason", "no pod spec"),
		)
		return nil
	}

	if podSpec.PodAgentPayload == nil || podSpec.PodAgentPayload.Spec == nil {
		outL.Info("skip boxes check",
			log.String("reason", "no podagent spec"),
		)
		return nil
	}

	for _, box := range podSpec.PodAgentPayload.Spec.Boxes {
		boxSpec := &check.BoxSpec{
			DeployUnit: deployUnit,
			Spec:       box,
			Pod:        podSpec.PodAgentPayload.Spec,
			Path: &xrayrpc.TargetPath{
				Name:   box.GetId(),
				Kind:   xrayrpc.TargetKind_TK_BOX,
				Parent: deployUnit.Path,
			},
		}

		if err := i.inspectBox(ctx, boxSpec, result); err != nil {
			return fmt.Errorf("box %q analysis fail: %w", targetPathID(boxSpec.Path), err)
		}
	}

	return nil
}

func (i *Inspector) inspectBox(ctx context.Context, box *check.BoxSpec, result *xrayrpc.AnalyzeResult) error {
	boxID := targetPathID(box.Path)
	appL := log.With(i.l, log.String("box_id", boxID))
	outL := log.With(i.outL, log.String("box_id", boxID))

	outL.Info("inspect box")
	for k := range i.checks[check.TargetKindBox] {
		ch := i.checks[check.TargetKindBox][k].(check.BoxCheck)
		issues, err := ch.CheckBox(outL, box)
		if err != nil {
			appL.Error("box check fail",
				log.String("check_name", ch.Name()),
				log.Error(err))
			outL.Error("box check fail",
				log.String("check_name", ch.Name()),
				log.Error(err))

			result.Warnings = append(result.Warnings, &xrayrpc.Warning{
				Target:  box.Path,
				Message: err.Error(),
			})
			continue
		}

		outL.Info("box check finished", log.String("check_name", ch.Name()))
		for _, issue := range issues {
			issue.Target = box.Path
			result.Issues = append(result.Issues, issue)
		}
	}

	if err := i.inspectBoxFS(ctx, box, result); err != nil {
		appL.Error("failed to check box fs: %w", log.Error(err))
		outL.Error("failed to check box fs: %w", log.Error(err))
	}

	boxRef := box.Spec.GetId()
	for _, spec := range box.Pod.Workloads {
		if spec.GetBoxRef() != boxRef {
			continue
		}

		wlSpec := &check.WorkloadSpec{
			Box:  box,
			Spec: spec,
			Path: &xrayrpc.TargetPath{
				Name:   spec.GetId(),
				Kind:   xrayrpc.TargetKind_TK_WORKLOAD,
				Parent: box.Path,
			},
		}

		if err := i.inspectWorkload(ctx, wlSpec, result); err != nil {
			return fmt.Errorf("workload %q analysis fail: %w", targetPathID(wlSpec.Path), err)
		}
	}

	return nil
}

func (i *Inspector) inspectBoxFS(ctx context.Context, box *check.BoxSpec, result *xrayrpc.AnalyzeResult) error {
	boxID := targetPathID(box.Path)
	appL := log.With(i.l, log.String("box_id", boxID))
	outL := log.With(i.outL, log.String("box_id", boxID))

	outL.Info("inspect box fs")
	outL.Info("prepare layers")

	rootfsURIs, err := i.resolveLayers(ctx, box, boxID)
	if err != nil {
		appL.Error("failed to resolve box rootfs", log.Error(err))
		outL.Error("failed to resolve box rootfs", log.Error(err))

		// that's fine
		return nil
	}

	if len(rootfsURIs) == 0 {
		outL.Error("box rootfs have no acceptable layers")
		outL.Error("box rootfs have no acceptable layers")

		// that's fine
		return nil
	}

	mountPointsToURIs, err := i.resolveStaticResources(ctx, box)
	if err != nil {
		appL.Error("failed to resolve box static resources", log.Error(err))
		outL.Error("failed to resolve box static resources", log.Error(err))

		// that's fine
		return nil
	}

	fsID := i.getFSId(rootfsURIs, mountPointsToURIs)
	appL.Info("FS id", log.String("fs_id", fsID))
	outL.Info("FS id", log.String("fs_id", fsID))

	var (
		volume        *porto.Volume
		cRequirements collector.ContainerRequirements
	)
	fsBuilt := false

	collectorsResults := make(map[string]*xrayrpc.CollectorResult)

	for k := range i.collectors[collector.TargetKindBoxFS] {
		recollect := false
		col := i.collectors[collector.TargetKindBoxFS][k].(collector.BoxFSCollector)
		colResults := new(xrayrpc.CollectorResult)
		rawCollectorResults, err := i.s3Storage.DownloadCollectorResults(col, fsID)

		if err == nil {
			outL.Info("got existing collector results", log.Error(err), log.String("collector_name", col.Name()), log.String("fs_id", fsID))
			appL.Info("got existing collector results", log.Error(err), log.String("collector_name", col.Name()), log.String("fs_id", fsID))

			if err = proto.Unmarshal(rawCollectorResults, colResults); err != nil {
				outL.Error("failed to parse collector results", log.Error(err), log.String("collector_name", col.Name()), log.String("fs_id", fsID))
				appL.Error("failed to parse collector results", log.Error(err), log.String("collector_name", col.Name()), log.String("fs_id", fsID))
				recollect = true
			}

		} else {
			outL.Warn("failed to download collector results", log.Error(err), log.String("collector_name", col.Name()), log.String("fs_id", fsID))
			appL.Warn("failed to download collector results", log.Error(err), log.String("collector_name", col.Name()), log.String("fs_id", fsID))
			recollect = true
		}

		if recollect {
			if !fsBuilt {
				outL.Info("no previous results, building FS", log.String("collector_name", col.Name()), log.String("fs_id", fsID))
				appL.Info("no previous results, building FS", log.String("collector_name", col.Name()), log.String("fs_id", fsID))
				fsBuilt = true
				volume, cRequirements = i.prepareFS(ctx, rootfsURIs, mountPointsToURIs, boxID)
				if volume == nil {
					return nil
				}
			}

			colResults, err = i.collectFromFS(ctx, boxID, fsID, col, volume, &cRequirements)
			if err != nil {
				colResults.Warnings = append(colResults.Warnings, &xrayrpc.FindingWarning{
					Id:      fsID,
					Message: err.Error(),
				})
			}
		}

		collectorsResults[col.Type()] = colResults
	}

	for k := range i.checks[check.TargetKindBoxFS] {
		ch := i.checks[check.TargetKindBoxFS][k].(check.BoxFSCheck)

		checkResults := new(xrayrpc.AnalyzeResult)
		outL.Info("start check",
			log.String("check_name", ch.Name()),
			log.String("collectors", strings.Join(ch.Collectors(), ", ")),
			log.String("fs_id", fsID),
		)
		appL.Info("start check",
			log.String("check_name", ch.Name()),
			log.String("collectors", strings.Join(ch.Collectors(), ", ")),
			log.String("fs_id", fsID),
		)
		err = haveNeededCollectors(ch, collectorsResults)
		if err == nil {
			checkIssues, err := ch.ProcessCollectorResults(i.outL, collectorsResults)
			if err != nil {
				outL.Error("check failed to parse results",
					log.Error(err),
					log.String("check_name", ch.Name()),
					log.String("collectors", strings.Join(ch.Collectors(), ", ")),
					log.String("fs_id", fsID),
				)
				appL.Error("check failed to parse results",
					log.Error(err),
					log.String("check_name", ch.Name()),
					log.String("collectors", strings.Join(ch.Collectors(), ", ")),
					log.String("fs_id", fsID),
				)
			}

			for _, issue := range checkIssues {
				issue.Target = box.Path
				checkResults.Issues = append(checkResults.Issues, issue)
			}
			for _, col := range ch.Collectors() {
				for _, warning := range collectorsResults[col].Warnings {
					checkResults.Warnings = append(checkResults.Warnings, &xrayrpc.Warning{
						Target:  box.Path,
						Message: warning.Message,
					})
				}
			}
			outL.Info("check finished",
				log.String("check_name", ch.Name()),
				log.String("collectors", strings.Join(ch.Collectors(), ", ")),
				log.String("fs_id", fsID),
			)
			appL.Info("check finished",
				log.String("check_name", ch.Name()),
				log.String("collectors", strings.Join(ch.Collectors(), ", ")),
				log.String("fs_id", fsID),
			)
		} else {
			outL.Error("no needed collector",
				log.Error(err),
				log.String("check_name", ch.Name()),
				log.String("collectors", strings.Join(ch.Collectors(), ", ")),
				log.String("fs_id", fsID),
			)
			appL.Error("no needed collector",
				log.Error(err),
				log.String("check_name", ch.Name()),
				log.String("collectors", strings.Join(ch.Collectors(), ", ")),
				log.String("fs_id", fsID),
			)
		}

		result.Issues = append(result.Issues, checkResults.Issues...)
		result.Warnings = append(result.Warnings, checkResults.Warnings...)
	}

	return nil
}

func (i *Inspector) inspectWorkload(_ context.Context, workload *check.WorkloadSpec, result *xrayrpc.AnalyzeResult) error {
	boxID := targetPathID(workload.Path)
	appL := log.With(i.l, log.String("workload_id", boxID))
	outL := log.With(i.outL, log.String("workload_id", boxID))

	outL.Info("inspect workload")
	for k := range i.checks[check.TargetKindWorkload] {
		ch := i.checks[check.TargetKindWorkload][k].(check.WorkloadCheck)
		issues, err := ch.CheckWorkload(outL, workload)
		if err != nil {
			appL.Error("workload check fail",
				log.String("check_name", ch.Name()),
				log.Error(err))
			outL.Error("workload check fail",
				log.String("check_name", ch.Name()),
				log.Error(err))

			result.Warnings = append(result.Warnings, &xrayrpc.Warning{
				Target:  workload.Path,
				Message: err.Error(),
			})
			continue
		}

		outL.Info("workload check finished", log.String("check_name", ch.Name()))
		for _, issue := range issues {
			issue.Target = workload.Path
			result.Issues = append(result.Issues, issue)
		}
	}

	return nil
}

func (i *Inspector) resolveLayers(ctx context.Context, box *check.BoxSpec, boxID string) ([]fetcher.URI, error) {
	resolverDocker := func() ([]string, error) {
		if box.DeployUnit.Spec.ImagesForBoxes == nil {
			// no docker images at all, that's fine
			return nil, nil
		}

		image, ok := box.DeployUnit.Spec.ImagesForBoxes[box.Spec.GetId()]
		if !ok {
			// current box doesn't use docker, that's fine
			return nil, nil
		}

		imageURI := fmt.Sprintf("%s/%s:%s", image.GetRegistryHost(), image.GetName(), image.GetTag())
		i.outL.Info("try to resolve docker image", log.String("uri", imageURI))
		imageInfo, err := dockinfo.Resolve(ctx, imageURI)
		if err != nil {
			return nil, fmt.Errorf("failed to resolve docker image (boxID=%s uri=%q): %w", boxID, imageURI, err)
		}

		result := make([]string, len(imageInfo.Layers))
		for k, layer := range imageInfo.Layers {
			i.outL.Info(
				"resolved docker layer",
				log.String("box_id", boxID),
				log.String("reference", layer.URI),
			)
			result[k] = layer.URI
		}
		return result, nil
	}

	resolvePorto := func() ([]string, error) {
		if box.Spec.Rootfs == nil || len(box.Spec.Rootfs.LayerRefs) == 0 {
			// no porto layers, that's fine
			return nil, nil
		}

		// porto layers
		i.outL.Info("try to resolve porto layers", log.String("box_id", boxID))
		result := make([]string, len(box.Spec.Rootfs.LayerRefs))
		for k, layerRef := range box.Spec.Rootfs.LayerRefs {
			found := false
			for _, layer := range box.Pod.Resources.Layers {
				if layer.GetId() == layerRef {
					i.outL.Info(
						"resolved layer reference",
						log.String("box_id", boxID),
						log.String("reference", layer.GetUrl()),
					)
					result[k] = layer.GetUrl()
					found = true
					break
				}
			}

			if !found {
				return nil, fmt.Errorf("can't find layer resource (boxID=%s): %s not found", boxID, layerRef)
			}
		}
		return result, nil
	}

	dockerLayers, err := resolverDocker()
	if err != nil {
		return nil, fmt.Errorf("failed to resolve docker image: %w", err)
	}

	portoLayers, err := resolvePorto()
	if err != nil {
		return nil, fmt.Errorf("failed to resolve porto layers: %w", err)
	}

	i.outL.Info("try to parse layers' URI", log.String("box_id", boxID))
	// docker image have bigger priority: https://st.yandex-team.ru/XRAY-1
	layers := append(dockerLayers, portoLayers...)
	layersURIs := make([]fetcher.URI, len(layers))

	for k, uri := range layers {
		parsedURI, err := i.layerStorage.ParseURI(ctx, uri)
		if err != nil {
			return nil, fmt.Errorf("failed to parse uri %q: %w", uri, err)
		}

		if parsedURI.Target != uri {
			i.l.Info("use normalized uri", log.String("orig_uri", uri), log.String("normalized_uri", parsedURI.Target))
			i.outL.Info("use normalized uri", log.String("orig_uri", uri), log.String("normalized_uri", parsedURI.Target))
		}

		layersURIs[k] = parsedURI
	}

	return layersURIs, nil
}

func (i *Inspector) importLayers(ctx context.Context, namespace string, layers []fetcher.URI) ([]string, error) {
	layersNames := make([]string, len(layers))
	layersCount := len(layers)

	for k, layerURI := range layers {
		i.outL.Debug("download layer", log.String("uri", layerURI.Target))
		layer, err := i.layerStorage.Download(ctx, layerURI)
		if err != nil {
			return nil, fmt.Errorf("failed to prepare layer (URI=%q): %w", layerURI, err)
		}

		i.cachedLayers = append(i.cachedLayers, layer)
		layerID := fmt.Sprintf("%s-%s-%s", LayerPrefix, namespace, layer.ID)
		importOpts := porto.ImportLayerOpts{
			Layer:   layerID,
			Tarball: layer.Path,
			Merge:   false,
		}

		i.outL.Debug(
			"import layer",
			log.String("namespace", namespace),
			log.String("layer_id", layerID),
			log.String("path", layer.Path),
		)

		if err := i.porto.ImportLayer(importOpts); err != nil {
			var portoErr *porto.Error
			if errors.As(err, &portoErr) && portoErr.Errno == layerAlreadyExistsErrno {
				// it's okay, layer already exists in porto
			} else {
				return nil, fmt.Errorf("failed to import layer %q: %w", layer.Path, err)
			}
		}

		layersNames[layersCount-1-k] = layerID
		i.portoLayers[layerID] = struct{}{}
	}

	return layersNames, nil
}

func (i *Inspector) buildVolume(volumeID string, layersNames []string, customMountPoints []string) (volume *porto.Volume, resultErr error) {
	volumePath := filepath.Join(i.workDir, "analyze-"+stringutil.ShaHex(volumeID))
	err := os.MkdirAll(volumePath, 0755)
	if err != nil {
		resultErr = fmt.Errorf("failed to prepare volume dir: %w", err)
		return
	}

	layers := strings.Join(layersNames, ";")
	i.outL.Debug(
		"build volume",
		log.String("layers", layers),
		log.String("path", volumePath),
	)

	volume, resultErr = i.porto.CreateVolume(volumePath, map[string]string{
		"layers":  layers,
		"backend": "overlay",
	})

	if resultErr != nil {
		resultErr = fmt.Errorf("failed to create volume: %w", resultErr)
		return
	}

	for _, name := range append(baseRootFSFolders, customMountPoints...) {
		targetPath := filepath.Join(volumePath, name)
		if _, err := os.Stat(targetPath); err == nil {
			continue
		}

		if err := os.MkdirAll(targetPath, 0755); err != nil {
			if err := volume.Destroy(); err != nil {
				i.outL.Error(
					"volume destroy fail",
					log.String("volume_id", volumeID),
					log.Error(err),
				)
			}

			return nil, fmt.Errorf("failed to create base folder %q: %w", targetPath, err)
		}
	}
	return
}

func (i *Inspector) resolveStaticResources(ctx context.Context, box *check.BoxSpec) (map[string]fetcher.URI, error) {
	if box.Pod.Resources == nil || len(box.Spec.StaticResources) == 0 {
		return nil, nil
	}

	resources := make(map[string]*podagent.TResource, len(box.Pod.Resources.StaticResources))
	for _, r := range box.Pod.Resources.StaticResources {
		resources[r.GetId()] = r
	}

	mountPointsToURIs := make(map[string]fetcher.URI)
	for _, r := range box.Spec.StaticResources {
		spec, ok := resources[r.GetResourceRef()]
		if !ok {
			i.outL.Warn("can't find static resource", log.String("ref", r.GetResourceRef()))
			continue
		}

		// TODO(anton-k): add files
		// TODO(anton-k): add sky_get
		if spec.Url == nil {
			i.outL.Warn("skip static resource w/o url", log.String("ref", r.GetResourceRef()))
			continue
		}

		mountPoint := r.GetMountPoint()
		if mountPoint == "" || strings.Contains(mountPoint, ";") || mountPoint[0] != '/' {
			i.outL.Warn("skip resource with invalid mount point", log.String("ref", r.GetResourceRef()), log.String("mount_point", r.GetMountPoint()))
			continue
		}

		resourceURL := spec.GetUrl()
		parsedURI, err := i.resourceStorage.ParseURI(ctx, resourceURL)
		if err != nil {
			return nil, fmt.Errorf("failed to parse uri %q: %w", resourceURL, err)
		}

		if parsedURI.Target != resourceURL {
			i.l.Info("use normalized uri", log.String("orig_uri", resourceURL), log.String("normalized_uri", parsedURI.Target))
			i.outL.Info("use normalized uri", log.String("orig_uri", resourceURL), log.String("normalized_uri", parsedURI.Target))
		}

		mountPointsToURIs[mountPoint] = parsedURI
	}

	return mountPointsToURIs, nil
}

func (i *Inspector) importStaticResources(ctx context.Context, mountPointsToURIs map[string]fetcher.URI) ([]string, error) {
	out := make([]string, 0, len(mountPointsToURIs))
	for mountPoint, rURI := range mountPointsToURIs {
		resource, err := i.resourceStorage.Download(ctx, rURI)
		if err != nil {
			// FIXME(anton-k): lost resource ref
			i.outL.Error("failed to download static resource", log.String("uri", rURI.Target), log.Error(err))
			continue
		}

		i.cachedResources = append(i.cachedResources, resource)
		out = append(out, fmt.Sprintf("%s %s ro", resource.Path, mountPoint))
	}

	return out, nil
}

func (i *Inspector) cleanupLayers() {
	i.outL.Debug("destroy layers")
	for layer := range i.portoLayers {
		if err := i.porto.RemoveLayer(layer); err != nil {
			i.l.Error("failed to destroy layer",
				log.String("layer", layer),
				log.Error(err),
			)
		}
	}

	i.outL.Debug("release layers")
	for _, l := range i.cachedLayers {
		l.Release()
	}
}

func (i *Inspector) cleanupResources() {
	i.outL.Debug("release resources")
	for _, l := range i.cachedResources {
		l.Release()
	}
}

func (i *Inspector) cleanupVolumes() {
	i.outL.Debug("destroy volumes")
	for _, v := range i.volumes {
		if err := v.Destroy(); err != nil {
			i.l.Error("failed to destroy volume",
				log.String("volume_path", v.Path()),
				log.Error(err),
			)
		}
	}
}

func (i *Inspector) getFSId(rootfsURIs []fetcher.URI, mountPointsToURIs map[string]fetcher.URI) string {
	layersHashes := make([]string, len(rootfsURIs))
	for j, uri := range rootfsURIs {
		layersHashes[j] = uri.ID
	}

	resourcesHashes := make([]string, 0, len(mountPointsToURIs))
	for mp, uri := range mountPointsToURIs {
		resourcesHashes = append(resourcesHashes, stringutil.ShaHex(mp, ":", uri.ID))

	}
	sort.Strings(resourcesHashes)

	layers := strings.Join(layersHashes, "|")
	resources := strings.Join(resourcesHashes, "|")

	return stringutil.ShaHex(layers, "|", resources)
}

func (i *Inspector) runContainer(containerName string, c container) (result *collector.ContainerResult, resultErr error) {
	startedAt := time.Now()
	_ = i.porto.Connection().Destroy(containerName) // destroy previously existed

	container, err := i.porto.CreateContainer(containerName)
	if err != nil {
		resultErr = err
		return
	}

	i.l.Debug("new container", log.String("name", containerName))

	defer func() {
		i.l.Debug("destroy container", log.String("name", container.GetName()))
		if err := container.Destroy(); err != nil {
			i.l.Warn("failed to destroy container", log.Error(err))
		}
	}()

	props := map[string]string{
		"command": c.cmd,
		// TODO(buglloc): shitty porto hack :/
		"root": c.root,
		"net":  c.PortoNet(),
		"bind": joinProps(c.Bind),
		"env":  joinProps(c.Env),
	}

	for opt, optVal := range i.portoOpts {
		switch opt {
		case "bind", "env":
			props[opt] = optVal + ";" + props[opt]
		default:
			props[opt] = optVal
		}
	}

	for propName, propVal := range props {
		err := container.SetProperty(propName, propVal)
		if err != nil {
			resultErr = err
			return
		}
	}

	err = container.SetProperty("root_readonly", "true")
	if err != nil {
		resultErr = err
		return
	}

	err = container.SetCapabilities(containerCapabilities)
	if err != nil {
		resultErr = err
		return
	}

	stdoutR, stdoutW, err := container.NewStdoutPipe()
	if err != nil {
		resultErr = fmt.Errorf("failed to create stdout pipes: %w", err)
		return
	}

	defer func() {
		_ = stdoutW.Close()
		_ = stdoutR.Close()
	}()

	stderrR, stderrW, err := container.NewStderrPipe()
	if err != nil {
		resultErr = fmt.Errorf("failed to create stderr pipes: %w", err)
		return
	}
	defer func() {
		_ = stderrW.Close()
		_ = stderrR.Close()
	}()

	i.l.Debug("start container", log.String("name", containerName))
	if err := container.Start(); err != nil {
		resultErr = err
		return
	}

	var stdWg sync.WaitGroup
	stdWg.Add(2)
	var (
		stdout    []byte
		stdoutErr error
	)
	go func() {
		stdout, stdoutErr = ioutil.ReadAll(stdoutR)
		if err := stdoutR.Close(); err != nil {
			i.l.Error("failed to close stdout pipe", log.Error(err))
		}
		stdWg.Done()
	}()

	var (
		stderr    []byte
		stderrErr error
	)
	go func() {
		stderr, stderrErr = ioutil.ReadAll(stderrR)
		if err := stderrR.Close(); err != nil {
			i.l.Error("failed to close stderr pipe", log.Error(err))
		}
		stdWg.Done()
	}()

	i.l.Debug("wait container", log.String("name", containerName))
	done, err := container.Wait(c.deadline)
	if err != nil {
		resultErr = err
		return
	}

	err = stdoutW.Close()
	if err != nil {
		resultErr = fmt.Errorf("failed to close stdout pipe: %w", err)
		return
	}

	err = stderrW.Close()
	if err != nil {
		resultErr = fmt.Errorf("failed to close stderr pipe: %w", err)
		return
	}

	i.l.Debug("wait container stdout/stderr", log.String("name", containerName))
	if !done {
		_ = container.StopWaitD(containerStopTimeout)
	}
	stdWg.Wait()

	if !done {
		resultErr = fmt.Errorf("container timeout exceed %s", c.deadline)
		return
	}

	if stdoutErr != nil {
		resultErr = fmt.Errorf("failed to read stdout pipe: %w", stdoutErr)
		return
	}

	if stderrErr != nil {
		resultErr = fmt.Errorf("failed to read stderr pipe: %w", stderrErr)
		return
	}

	status, err := container.GetProperty("exit_status")
	if err != nil {
		resultErr = fmt.Errorf("failed to get container exit_status: %w", err)
		return
	}
	uStatus, _ := strconv.ParseUint(status, 10, 32)

	i.l.Debug("container finished", log.String("name", containerName))
	return &collector.ContainerResult{
		Status:  syscall.WaitStatus(uint32(uStatus)),
		Stdout:  stdout,
		Stderr:  bytes.TrimSpace(stderr),
		Elapsed: time.Since(startedAt),
	}, nil
}

func (i *Inspector) collectFromFS(ctx context.Context, boxID, fsID string, col collector.BoxFSCollector,
	volume *porto.Volume, cRequirements *collector.ContainerRequirements) (*xrayrpc.CollectorResult, error) {
	colResults := new(xrayrpc.CollectorResult)
	appL := log.With(i.l, log.String("box_id", boxID))
	outL := log.With(i.outL, log.String("box_id", boxID))

	containerName := fmt.Sprintf(
		"self/xray-runtime-%s-%s-%s-%s",
		col.Type(), col.Version(), i.analyzeID, stringutil.ShaHex(boxID),
	)

	outL.Info("start collector",
		log.String("collector_name", col.Name()),
		log.String("container_name", containerName))

	// ND! "self/" prefix for porto container matter. This mean starts really child container
	collectorContainerResult, err := i.runContainer(
		containerName,
		container{
			cmd:                   col.Command(),
			deadline:              col.Deadline(),
			root:                  volume.Path(),
			ContainerRequirements: collector.MergeContainerRequirements(col.Requirements(), *cRequirements),
		},
	)

	if collectorContainerResult != nil {
		outL.Info("collector finished", log.String("collector_name", col.Name()))
	}

	if err != nil {
		appL.Warn("collector failed",
			log.String("collector_name", col.Name()),
			log.Error(err))
		outL.Warn("collector failed",
			log.String("collector_name", col.Name()),
			log.Error(err))

		return colResults, err
	}

	collectorFinding, err := col.ProcessContainerResult(i.outL, collectorContainerResult, fsID)
	if err != nil {
		appL.Error("failed to process collector output",
			log.String("collector_name", col.Name()),
			log.Error(err))
		outL.Error("failed to process collector output",
			log.String("collector_name", col.Name()),
			log.Error(err))

		return colResults, err
	}

	outL.Info("collector output processed", log.String("collector_name", col.Name()))
	colResults.Finding = collectorFinding

	data, err := proto.Marshal(colResults)
	if err != nil {
		outL.Error("failed to upload collector results: encode failed", log.Error(err), log.String("collector_name", col.Name()))
		appL.Error("failed to upload collector results: encode failed", log.Error(err), log.String("collector_name", col.Name()))
	}
	err = i.s3Storage.UploadCollectorResults(ctx, col, fsID, bytes.NewReader(data))
	if err != nil {
		outL.Error("failed to upload collector results", log.Error(err), log.String("collector_name", col.Name()))
		appL.Error("failed to upload collector results", log.Error(err), log.String("collector_name", col.Name()))
	}
	// don't return error if failed to upload collector results
	return colResults, nil
}

func (i *Inspector) prepareFS(ctx context.Context, rootfsURIs []fetcher.URI, mountPointsToURIs map[string]fetcher.URI, boxID string) (*porto.Volume, collector.ContainerRequirements) {
	appL := log.With(i.l, log.String("box_id", boxID))
	outL := log.With(i.outL, log.String("box_id", boxID))

	customMountPoints := make([]string, 0, len(mountPointsToURIs))
	for mountPoint := range mountPointsToURIs {
		customMountPoints = append(customMountPoints, mountPoint)
	}

	binds, err := i.importStaticResources(ctx, mountPointsToURIs)
	if err != nil {
		appL.Error("failed to import box static resources", log.Error(err))
		outL.Error("failed to import box static resources", log.Error(err))

		// that's fine
		return nil, collector.ContainerRequirements{}
	}

	outL.Info("import layers")
	layers, err := i.importLayers(ctx, i.analyzeID, rootfsURIs)
	if err != nil {
		appL.Error("failed to import box rootfs layers", log.Error(err))
		outL.Error("failed to import box rootfs layers", log.Error(err))

		// that's fine
		return nil, collector.ContainerRequirements{}
	}

	volumeID := fmt.Sprintf("volume-%s-%s", i.analyzeID, stringutil.ShaHex(boxID))
	outL.Info("build rootfs", log.String("volume_id", volumeID))
	volume, err := i.buildVolume(volumeID, layers, customMountPoints)
	if err != nil {
		appL.Error("failed to build box rootfs", log.String("volume_id", volumeID), log.Error(err))
		outL.Error("failed to build box rootfs", log.String("volume_id", volumeID), log.Error(err))

		// that's fine
		return nil, collector.ContainerRequirements{}
	}
	i.volumes = append(i.volumes, volume)

	return volume, collector.ContainerRequirements{
		Bind: binds,
	}
}

func haveNeededCollectors(check check.BoxFSCheck, collectorsResults map[string]*xrayrpc.CollectorResult) error {
	for _, col := range check.Collectors() {
		if _, ok := collectorsResults[col]; !ok {
			return fmt.Errorf("no collector %s", col)
		}
	}
	return nil
}
