package worker

import (
	"context"
	"encoding/base64"
	"fmt"
	"os"
	"path"
	"path/filepath"
	"runtime/debug"
	"time"

	"github.com/aws/aws-sdk-go/service/sqs"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/queue"
	"a.yandex-team.ru/security/xray/internal/servers/worker/inspect"
	"a.yandex-team.ru/security/xray/internal/servers/worker/results"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/yp/go/yp"
	"a.yandex-team.ru/yp/go/yson/ypapi"
)

type (
	processRequest struct {
		analyzeID  string
		workDir    string
		appLogger  log.Logger
		procLogger log.Logger
		msg        string
	}
)

func (w *Worker) processMessage(msg *sqs.Message) {
	defer func() {
		if r := recover(); r != nil {
			w.log.Error("panic while process message", log.Any("err", r), log.ByteString("stack", debug.Stack()))
		}
	}()

	ctx, cancel := context.WithTimeout(context.Background(), w.analyzeTimeout)
	defer cancel()

	analyzeID := *msg.MessageId

	// Setup worker logger
	appLog := log.With(w.log, log.String("analyze_id", analyzeID))

	startTS := time.Now()

	// hostname is a hack due to shitty Y.Deploy log rendering
	hostname, _ := os.Hostname()
	appLog.Info("incoming message", log.String("hostname", hostname))
	defer func() {
		appLog.Info("message processed", log.Duration("elapsed", time.Since(startTS)))
	}()

	// Setup analyze directory
	analyzeDir := filepath.Join(w.cfg.Worker.WorkDir, analyzeID)
	if err := os.MkdirAll(analyzeDir, 0777); err != nil {
		appLog.Error("failed to create analyze directory", log.Error(err))
		logUUID, _ := uuidGen.NewV4()
		analyzeDir = path.Join(w.cfg.Worker.WorkDir, "fallback-"+logUUID.String())
		_ = os.MkdirAll(analyzeDir, 0777)
	}
	appLog.Debug("analyze directory created", log.String("path", analyzeDir))

	// Setup analysis processLogger
	logPath := path.Join(analyzeDir, "analyze.log")
	procLog, err := w.newProcessorLogger(logPath)
	if err != nil {
		appLog.Error("failed to create processor logger", log.Error(err))
		return
	}

	// Analyze it!
	response := w.processRawRequest(ctx, processRequest{
		analyzeID:  analyzeID,
		workDir:    analyzeDir,
		appLogger:  appLog,
		procLogger: procLog,
		msg:        *msg.Body,
	})

	if err = w.splunk.AnalysisComplete(time.Since(startTS), response); err != nil {
		appLog.Error("failed to send results into slunk", log.Error(err))
	}

	// Deal with processLogger
	if err := procLog.L.Sync(); err != nil {
		appLog.Error("failed to close analysis log file", log.Error(err))
	}

	if uploadedLog, err := w.uploadLog(ctx, analyzeID, logPath); err == nil {
		response.LogPath = uploadedLog
		appLog.Debug("log uploaded", log.String("path", uploadedLog))
	} else {
		appLog.Error("failed to upload analysis log file", log.Error(err))
	}

	if uploadedResult, err := w.uploadResult(ctx, analyzeID, response); err == nil {
		response.ResultPath = uploadedResult
		appLog.Debug("result uploaded", log.String("path", uploadedResult))
	} else {
		appLog.Error("failed to upload results", log.Error(err))
		return
	}

	err = w.saveResponse(ctx, response)
	if err != nil {
		appLog.Error("failed to save response",
			log.String("analyze_id", analyzeID),
			log.Error(err),
		)
		return
	}
	appLog.Debug("result saved")

	// clean up
	err = os.RemoveAll(analyzeDir)
	if err != nil {
		appLog.Error("failed to delete analyze directory",
			log.String("path", analyzeID),
			log.Error(err),
		)
	}

	err = w.queue.DeleteMessage(ctx, &queue.DeleteOptions{
		QueueURL:      w.cfg.RequestsQueueURL(),
		ReceiptHandle: msg.ReceiptHandle,
	})
	if err != nil {
		appLog.Error("failed to delete message",
			log.String("receipt_handle", *msg.ReceiptHandle),
			log.Error(err),
		)
	}
}

func (w *Worker) processRawRequest(ctx context.Context, req processRequest) *results.Analyze {
	req.appLogger.Info("start analysis")
	req.procLogger.Infof("start analysis %q", req.analyzeID)

	startTS := time.Now()
	defer func() {
		req.procLogger.Info("finish analysis", log.Duration("elapsed", time.Since(startTS)))
		req.appLogger.Info("finish analysis")
	}()

	response := &results.Analyze{
		AnalyzeID: req.analyzeID,
	}

	rawMessage, err := base64.RawStdEncoding.DecodeString(req.msg)
	if err != nil {
		req.appLogger.Error("failed to parse request",
			log.Error(err),
		)

		response.Status = xrayrpc.AnalysisStatusKind_ASK_FAIL
		response.StatusDescription = "failed to parse analyze request"
		return response
	}

	var analyzeRequest xrayrpc.InternalAnalyzeRequest
	if err = proto.Unmarshal(rawMessage, &analyzeRequest); err != nil {
		req.appLogger.Error("failed to decode analyze request",
			log.Error(err),
		)

		response.Status = xrayrpc.AnalysisStatusKind_ASK_FAIL
		response.StatusDescription = "failed to decode analyze request"
		return response
	}

	response.Stage = analyzeRequest.Stage
	req.procLogger.Info("process analysis")

	stageInfo, err := w.fetchStageInfo(ctx, analyzeRequest.Stage.Id)
	if err != nil {
		req.procLogger.Error("failed to get stage info", log.Error(err))
		req.appLogger.Error("failed to get stage info", log.String("stage_id", analyzeRequest.Stage.Id), log.Error(err))

		response.Status = xrayrpc.AnalysisStatusKind_ASK_FAIL
		response.StatusDescription = fmt.Sprintf("prepare stage error: %s", err.Error())
		return response
	}

	if stageInfo.GetMeta().GetUuid() != analyzeRequest.Stage.Uuid {
		response.Status = xrayrpc.AnalysisStatusKind_ASK_ABORT
		response.StatusDescription = fmt.Sprintf(
			"prepare stage error: stage UUID mismatch %s (cur) != %s (requested)",
			stageInfo.GetMeta().GetUuid(), analyzeRequest.Stage.Uuid,
		)
		return response
	}

	if stageInfo.GetSpec().GetRevision() != analyzeRequest.Stage.Revision {
		response.Status = xrayrpc.AnalysisStatusKind_ASK_ABORT
		response.StatusDescription = fmt.Sprintf(
			"prepare stage error: stage revision mismatch %d (cur) != %d (requested)",
			stageInfo.GetSpec().GetRevision(), analyzeRequest.Stage.Revision,
		)
		return response
	}

	inspector := inspect.NewInspector(
		inspect.WithLayerStorage(w.layerStorage),
		inspect.WithResourcesStorage(w.resStorage),
		inspect.WithS3Storage(w.s3Storage),
		inspect.WithPortoAPI(w.porto),
		inspect.WithChecks(w.checks),
		inspect.WithCollectors(w.collectors),
		inspect.WithLogger(req.appLogger),
		inspect.WithOutputLogger(req.procLogger),
		inspect.WithAnalyzeID(req.analyzeID),
		inspect.WithWorkDir(req.workDir),
		inspect.WithPortoOpts(w.cfg.Worker.PortoOpts),
	)
	defer inspector.Close()

	response.Results, err = inspector.Inspect(ctx, stageInfo)
	if err == nil {
		response.Status = xrayrpc.AnalysisStatusKind_ASK_DONE
	} else {
		req.procLogger.Error("failed to process stage", log.Error(err))
		response.Status = xrayrpc.AnalysisStatusKind_ASK_FAIL
		response.StatusDescription = fmt.Sprintf("process stage error: %s", err.Error())
	}

	return response
}

func (w *Worker) fetchStageInfo(ctx context.Context, stageID string) (*ypapi.TStage, error) {
	rsp, err := w.yp.GetStage(
		ctx,
		yp.GetStageRequest{
			ID:        stageID,
			Format:    yp.PayloadFormatYson,
			Selectors: []string{"/spec", "/meta"},
		},
	)

	if err != nil {
		return nil, fmt.Errorf("get fail: %w", err)
	}

	stageInfo := &ypapi.TStage{
		Spec: new(ypapi.TStageSpec),
		Meta: new(ypapi.TStageMeta),
	}
	if err = rsp.Fill(stageInfo.Spec, stageInfo.Meta); err != nil {
		return nil, fmt.Errorf("fill fail: %w", err)
	}

	return stageInfo, nil
}
