package splunk

import (
	"bytes"
	"crypto/tls"
	"encoding/json"
	"errors"
	"fmt"
	"strings"
	"sync"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/xray/internal/servers/worker/results"
	"a.yandex-team.ru/security/xray/pkg/checks"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	hecEndpoint  = "https://hatch.yandex.net/services/collector"
	hecRetries   = 3
	queueTimeout = 5 * time.Second
	queueSize    = 200
)

var _ Sender = (*HecSender)(nil)

type HecSender struct {
	indexName  string
	sourceType string
	httpc      *resty.Client
	l          log.Logger
	queue      chan Results
	wg         sync.WaitGroup
	buf        bytes.Buffer
}

func NewHecSender(opts ...Option) *HecSender {
	s := &HecSender{
		l: &nop.Logger{},
		httpc: resty.New().
			SetRetryCount(hecRetries).
			SetHeader("Content-Type", "application/json"),
		queue: make(chan Results, queueSize),
	}

	for _, opt := range opts {
		opt(s)
	}

	certPool, err := certifi.NewCertPool()
	if err != nil {
		s.l.Error("splunk: failed to configure TLS cert pool", log.Error(err))
	} else {
		s.httpc.SetTLSClientConfig(&tls.Config{RootCAs: certPool})
	}

	return s
}

func (s *HecSender) Start() error {
	s.wg.Add(1)
	go func() {
		defer s.wg.Done()

		send := func(analysisID string, body []byte) {
			rsp, err := s.httpc.R().
				SetBody(body).
				Post(hecEndpoint)

			if err != nil {
				s.l.Error("splunk: send failed",
					log.String("analysis_id", analysisID),
					log.Error(err),
				)
				return
			}

			if !rsp.IsSuccess() {
				s.l.Error("splunk: send failed",
					log.String("analysis_id", analysisID),
					log.String("error",
						fmt.Sprintf("non 200 status code: %d", rsp.StatusCode()),
					),
				)
				return
			}
		}

		for result := range s.queue {
			id := result.Results.AnalyzeID
			data, err := s.resultsToEvents(result)
			if err != nil {
				s.l.Error("splunk: failed to encode result", log.String("analysis_id", id))
				continue
			}

			send(id, data)
		}
	}()

	return nil
}

func (s *HecSender) Stop() error {
	close(s.queue)
	s.wg.Wait()
	return nil
}

func (s *HecSender) AnalysisComplete(timeSpent time.Duration, result *results.Analyze) error {
	select {
	case s.queue <- Results{
		Time:    time.Now(),
		Spent:   timeSpent,
		Results: result,
	}:
		return nil
	case <-time.After(queueTimeout):
		return errors.New("splunk event queue timeout")
	}
}

func (s *HecSender) resultsToEvents(results Results) ([]byte, error) {
	encoder := json.NewEncoder(&s.buf)
	defer s.buf.Reset()

	var err error
	timestamp := results.Time.Unix()
	analysisResults := results.Results
	baseEvent := BaseDataEvent{
		AnalysisID: analysisResults.AnalyzeID,
	}

	if analysisResults.Stage != nil {
		baseEvent.StageID = analysisResults.Stage.Id
		baseEvent.StageUUID = analysisResults.Stage.Uuid
		baseEvent.StageRevision = analysisResults.Stage.Revision
	}

	if analysisResults.Results != nil {
		for _, warning := range analysisResults.Results.Warnings {
			err = encoder.Encode(Event{
				Timestamp:  timestamp,
				SourceType: s.sourceType,
				Index:      s.indexName,
				Data: Warning{
					BaseDataEvent: baseEvent,
					EventType:     "warning",
					Target:        targetPathID(warning.Target),
					Message:       warning.Message,
				},
			})
			if err != nil {
				return nil, err
			}
		}

		for _, issue := range analysisResults.Results.Issues {
			issueType, splunkIssue := checks.ExtractDetails(issue.Details)
			err = encoder.Encode(Event{
				Timestamp:  timestamp,
				SourceType: s.sourceType,
				Index:      s.indexName,
				Data: Issue{
					BaseDataEvent: baseEvent,
					EventType:     "issue",
					IssueKind:     issue.Kind.String(),
					IssueType:     issueType,
					Severity:      issue.Severity.String(),
					Target:        targetPathID(issue.Target),
					Details:       splunkIssue,
				},
			})
			if err != nil {
				return nil, err
			}
		}
	}

	err = encoder.Encode(
		Event{
			Timestamp:  timestamp,
			SourceType: s.sourceType,
			Index:      s.indexName,
			Data: AnalysisComplete{
				BaseDataEvent:     baseEvent,
				EventType:         "analysis_complete",
				Status:            analysisResults.Status.String(),
				StatusDescription: analysisResults.StatusDescription,
				Duration:          results.Spent.Seconds(),
			},
		},
	)
	if err != nil {
		return nil, err
	}

	return s.buf.Bytes(), nil
}

func targetPathID(targetPath *xrayrpc.TargetPath) string {
	var path []string
	parent := targetPath
	for ; parent != nil; parent = parent.Parent {
		path = append([]string{parent.Name}, path...)
	}

	return strings.Join(path, ".")
}
