package fetcher

import (
	"context"
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/libs/go/yahttp"
	"a.yandex-team.ru/security/xray/internal/sandbox"
	"a.yandex-team.ru/security/xray/internal/stringutil"
)

const (
	MaxSize      = 5 * 1024 * 1024 * 1024
	FetchTimeout = 5 * time.Minute
)

type Fetcher struct {
	sandboxc       *sandbox.Client
	httpc          *http.Client
	singleFileMode bool
	log            log.Logger
	maxSize        int64
}

func NewFetcher(opts ...Option) *Fetcher {
	out := &Fetcher{
		httpc: yahttp.NewClient(yahttp.Config{
			RedirectPolicy: yahttp.RedirectNoFollow,
			DialTimeout:    1 * time.Second,
			Timeout:        FetchTimeout,
		}),
		log:     &nop.Logger{},
		maxSize: MaxSize,
	}

	for _, opt := range opts {
		opt(out)
	}

	out.sandboxc = sandbox.New(out.log)
	return out
}

func (f *Fetcher) ParseURI(ctx context.Context, uri string) (URI, error) {
	parsedURI, err := url.Parse(uri)
	if err != nil {
		return URI{}, err
	}

	resourceToTorrent := func(resourceID string) (string, error) {
		resourceInfo, err := f.sandboxc.GetResourceInfo(ctx, resourceID)
		if err != nil {
			f.log.Error("failed to get sb-resource info",
				log.String("resource_id", resourceID),
				log.String("uri", uri),
				log.Error(err),
			)

			return "", fmt.Errorf("failed to get SB-resource %s info: %w", resourceID, err)
		}

		if resourceInfo.SkynetID == "" {
			return "", fmt.Errorf("no SkynetID for SB-resource: %s", resourceID)
		}

		if !strings.HasPrefix(resourceInfo.SkynetID, "rbtorrent:") {
			return "", fmt.Errorf("invalid SkynetID for SB-resouce %s: %s", resourceID, resourceInfo.SkynetID)
		}

		return resourceInfo.SkynetID, nil
	}

	uriToLayer := func(parsedURI *url.URL) (URI, error) {
		fallback := false
		switch parsedURI.Scheme {
		case "http", "https":
			if parsedURI.Host != sandbox.ProxyHost {
				return URI{Kind: URIKindHTTP, Target: uri}, nil
			}

			// this is proxy resource, try to handle it pro pertly: XRAY-11
			fallback = true
			parsedURI.Opaque = strings.Trim(parsedURI.Path, "/")
			fallthrough
		case "sbr":
			// XRAY-30
			torrentURI, err := resourceToTorrent(parsedURI.Opaque)
			if err != nil {
				if fallback {
					// fallback to http, this is fine
					return URI{Kind: URIKindHTTP, Target: uri}, nil
				}
				return URI{}, err
			}

			uri = torrentURI
			fallthrough
		case "rbtorrent":
			return URI{Kind: URIKindSkynet, Target: uri}, nil
		default:
			return URI{}, fmt.Errorf("unsupported uri scheme: %s", parsedURI.Scheme)
		}
	}
	out, err := uriToLayer(parsedURI)
	if err != nil {
		return URI{}, err
	}

	out.ID = stringutil.ShaHex(uri)
	return out, nil
}

func (f *Fetcher) Download(ctx context.Context, uri URI, dst string) (int64, error) {
	var size int64
	var err error
	switch uri.Kind {
	case URIKindHTTP:
		size, err = f.httpDownload(ctx, uri, dst)
	case URIKindSkynet:
		size, err = f.skynetDownload(ctx, uri, dst)
	default:
		err = fmt.Errorf("unsupported layer uri of kind %d: %s", uri.Kind, uri.Target)
	}

	if err != nil {
		return 0, fmt.Errorf("failed to download layer (kind=%d, uri='%s'): %w", uri.Kind, uri.Target, err)
	}
	return size, nil
}

func (f *Fetcher) Close() {
	f.sandboxc.Close()
}
